/*******************************************************************************
 * Copyright (c) 2018, 2023 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl.query;

import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.serialization.Serializer;
import org.simantics.db.ObjectResourceIdMap;
import org.simantics.db.common.WriteBindings;
import org.simantics.db.common.exception.DebugException;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.impl.graph.ReadGraphImpl;
import org.simantics.db.impl.procedure.InternalProcedure;
import org.simantics.db.service.CollectionSupport;

public final class ChildMap extends UnaryQueryP<ObjectResourceIdMap<String>> {

    public ChildMap(final int r) {
        super(r);
    }

    static final ChildMap entry(final QueryProcessor provider, final int r) {
        return (ChildMap)provider.cache.childMapMap.get(r);
    }

    @Override
    public final void removeEntry(QueryProcessor provider) {
        provider.cache.remove(this);
    }

    @Override
    public void compute(ReadGraphImpl graph, final InternalProcedure<ObjectResourceIdMap<String>> procedure)
            throws DatabaseException {
        computeForEach(graph, id, this, procedure);
    }

    public static void computeForEach(ReadGraphImpl graph, final int root, final ChildMap entry,
            final InternalProcedure<ObjectResourceIdMap<String>> procedure_) throws DatabaseException {

        InternalProcedure<ObjectResourceIdMap<String>> procedure = entry != null ? entry : procedure_;

        computeForEach2(graph, root, entry, procedure);

        if (entry != null)
            entry.performFromCache(graph, procedure_);

    }

    public static void computeForEach2(ReadGraphImpl graph, final int root, final ChildMap parent,
            final InternalProcedure<ObjectResourceIdMap<String>> procedure) throws DatabaseException {

        if (root == 0) {
            procedure.execute(graph, null);
            return;
        }

        QueryProcessor processor = graph.processor;

        final int consistsOf = processor.getConsistsOf();
        final int hasName = processor.getHasName();

        ObjectResourceIdMap<String> result = graph.getService(CollectionSupport.class)
                .createObjectResourceMap(String.class);

        QueryCache.runnerObjects(graph, root, consistsOf, parent, null, new SyncIntProcedure() {

            @Override
            public void run(ReadGraphImpl graph) throws DatabaseException {
                procedure.execute(graph, result);
            }

            @Override
            public void finished(ReadGraphImpl graph) throws DatabaseException {
                dec(graph);
            }

            @Override
            public void execute(ReadGraphImpl graph, final int obj) throws DatabaseException {

                inc();

                QueryCache.runnerObjects(graph, obj, hasName, parent, null, new IntProcedure() {

                    @Override
                    public void execute(ReadGraphImpl graph, int i) throws DatabaseException {

                        inc();

                        QueryCache.runnerValueQuery(graph, i, parent, null, new InternalProcedure<byte[]>() {

                            @Override
                            public void execute(ReadGraphImpl graph, byte[] value) throws DatabaseException {

                                if (value != null) {

                                    try {

                                        Binding b = WriteBindings.STRING;
                                        Serializer serializer = b.serializer();
                                        final String part = (String) serializer.deserialize(value);
                                        result.putId(part, obj);

                                    } catch (Throwable e) {
                                        if (DebugException.DEBUG)
                                            new DebugException(e).printStackTrace();
                                    }

                                }

                                dec(graph);

                            }

                            @Override
                            public void exception(ReadGraphImpl graph, Throwable t) throws DatabaseException {
                                dec(graph);
                            }

                        });

                    }

                    @Override
                    public void finished(ReadGraphImpl graph) throws DatabaseException {
                        dec(graph);
                    }

                    @Override
                    public void exception(ReadGraphImpl graph, Throwable t) throws DatabaseException {
                        dec(graph);
                    }

                });

            }

        });

    }

    @Override
    public void serializeValue(QuerySerializer serializer) {
        ObjectResourceIdMap<String> is = getResult();
        serializer.writeLE(is.size());
        for(String s : is.keySet()) {
            serializer.addString(s);
            serializer.addResource(is.getId(s));
        }
    }

    @Override
    public String toString() {
        return "ChildMap[" + id + "]";
    }

}
