/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.common.uri;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.util.URIStringUtils;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ResourceRead;
import org.simantics.db.exception.AssumptionException;
import org.simantics.db.exception.DatabaseException;
import org.simantics.layer0.Layer0;

/**
 * Tries to retrieve URIs for specified resources. Throws
 * {@link AssumptionException} if URI is undefined. This query follows the
 * semantics of {@link ReadGraph#getURI(Resource)}.
 * 
 * @author Antti Villberg
 */
public class ResourceToURI extends ResourceRead<String> {

    public ResourceToURI(Resource resource) {
        super(resource);
    }

    @Override
    public String perform(ReadGraph graph) throws DatabaseException {
        
        if (resource.equals(graph.getRootLibrary())) return "http:/";
        
        Layer0 L0 = Layer0.getInstance(graph);
        String name = graph.getPossibleRelatedValue(resource, L0.HasName, Bindings.STRING);
        if (name == null) throw new AssumptionException("resource " + resource + " does not have a name");
        
        String escapedName = URIStringUtils.escape((String) name);
        
        Resource parent = graph.getPossibleObject(resource, L0.PartOf);
        if(parent == null) throw new AssumptionException("resource " + resource + " does not have a parent");
        
        String parentURI = graph.syncRequest(new ResourceToURI(parent));
        if(parentURI == null) throw new AssumptionException("parent resource " + parent + " has no URI");

        return parentURI + "/" + escapedName;
        
    }

}
