/*******************************************************************************
 * Copyright (c) 2007 VTT Technical Research Centre of Finland and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.util;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.Datatypes;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.NumberBinding;
import org.simantics.databoard.binding.StringBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.error.RuntimeBindingException;
import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.databoard.file.RuntimeIOException;
import org.simantics.databoard.serialization.Serializer;
import org.simantics.databoard.serialization.SerializerConstructionException;

public class StringUtil {

	/**
	 * Convert variant to string
	 * 
	 * @param v
	 * @return string representation
	 */
	public static String variantToString(Variant v) {
		try {
			return (String) Bindings.STR_VARIANT.create(v.getBinding(), v.getValue());
		} catch (BindingException e) {
			throw new RuntimeBindingException(e);
		}
	}
	
	/**
	 * Convert variant's string representation to variant
	 * 
	 * @param str
	 * @return variant
	 * @throws BindingException
	 */
	public static Variant stringToVariant(String str) throws BindingException {
		return (Variant) Bindings.VARIANT.create(Bindings.STR_VARIANT, str);
	}
	
	/**
	 * Convert variant to nice user readable string.
	 * The string is not convertible back to variant without type information. 
	 * 
	 * @param v
	 * @return string representation of the variant
	 */
	public static String variantToUserReadableString(Variant v) {
		try {
			Binding binding = v.getBinding();
			Object value = v.getValue();
			if ( binding == Bindings.STRING ) return (String) value;
			if ( binding.type().equals(Datatypes.STRING)) {
				StringBinding sb = (StringBinding) binding;
				return sb.getValue(value);
			}
		
			if (binding instanceof NumberBinding) {
				return binding.toString(value);
			}
			
			Serializer s = Bindings.getSerializer( Bindings.VARIANT );
			byte[] bytes = s.serialize( v );
			return Base64.encodeBytes(bytes, Base64.URL_SAFE);
		} catch (BindingException e) {
			// UNEXPECTED
			throw new RuntimeBindingException(e);
		} catch (SerializerConstructionException e) {
			throw new RuntimeException(e);
		} catch (IOException e) {
			throw new RuntimeIOException(e);
		}
	}    
	
	/**
	 * Create escaped filename from a string
	 * 
	 * @param name any string
	 * @return file compatible string
	 */
	public static String escapeToFileName(String name) {
		try {
			return java.net.URLEncoder.encode(name, "UTF-8");
		} catch (UnsupportedEncodingException e) {
			// never expected
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * Unescape filename into string 
	 * 
	 * @param filename 
	 * @return any string
	 */
	public static String unescapeFileName(String filename) {
		try {
			return java.net.URLDecoder.decode(filename, "UTF-8");
		} catch (UnsupportedEncodingException e) {
			// never expected
			throw new RuntimeException(e);
		}
	}
	
    public static boolean simplePatternMatch(String str, String simplePattern)
    {
        try {
            Pattern ptr = compileSimplePattern(simplePattern);
            Matcher m = ptr.matcher(str);
            return m.matches();
        } catch (PatternSyntaxException pse) {
            return false;
        }
    }
    
    /**
     * Compiles pattern from simple pattern. Simple pattern is normal
     * wild card compare that supports * and ? wild cards.
     * 
     * @param patternStr simple pattern
     * @return Regexp pattern
     */
    public static Pattern compileSimplePattern(String patternStr)
    throws PatternSyntaxException
    {
        String str ="";
        for (int i=0; i<patternStr.length(); i++)
        {
            char c = patternStr.charAt(i);
            if ( (c>='a'&&c<='z') || (c>='A'&&c<='Z') || (c>='0'&&c<='9'))
                str += c;
            else if ( c=='?' )
                str += ".?";
            else if ( c=='*' )
                str += ".*";
            else str += "\\"+c;
        }
        return Pattern.compile(str);
    }    
		
}

