package org.simantics.databoard.primitives;

import java.math.BigInteger;


/**
 * Unsigned immutable 64-bit integer value. The value is between 0 and 18446744073709551615.
 *
 * @author Toni Kalajainen <toni.kalajainen@iki.fi>
 */
public abstract class UnsignedLong extends Number implements Comparable<Number> {

	static final long serialVersionUID = 1L;
	static final long L_MIN_VALUE = 0;
	static final BigInteger HALF = new BigInteger("9223372036854775808"); // 2^63
	static final BigInteger BI_MIN_VALUE = new BigInteger("0");
	static final BigInteger BI_MAX_VALUE = new BigInteger("18446744073709551615"); // 2^64-1
	
	public static final UnsignedLong MIN_VALUE, MAX_VALUE, ZERO;
	

	/** Value container */
	long value;
	
	/**
	 * Get cached or create new immutable unsigned long
	 *  
	 * @param value
	 * @return immutable unsigned long
	 */
    public static UnsignedLong valueOf(long value) {
		if ( value>=0 && value<CACHE.length ) return CACHE[(int)value];
    	if ( value<L_MIN_VALUE ) throw new IllegalArgumentException("Argument is not within range");
		return new UnsignedLong.Immutable(value);    	
    }    
	
    /**
     * Get cached or create new immutable unsigned long
     * 
     * @param bits long bits
     * @return immutable unsigned long
     */
    public static UnsignedLong fromBits(long bits) {
		if (bits>=0 && bits<CACHE.length) return CACHE[(int)bits];
    	UnsignedLong result = new UnsignedLong.Immutable();    	
    	result.value = bits;
    	return result;
    }
    
	public static class Mutable extends UnsignedLong {
	
		private static final long serialVersionUID = 1L;

		Mutable() {}
		
		public Mutable(int value) throws IllegalArgumentException {
	    	if ( value<L_MIN_VALUE ) throw new IllegalArgumentException("Argument is not within range");
	        this.value = value;
	    }

	    public Mutable(long value) throws IllegalArgumentException {
	    	if ( value<L_MIN_VALUE ) throw new IllegalArgumentException("Argument is not within range");
	        this.value = (int) value;
	    }

	    public Mutable(BigInteger value) throws IllegalArgumentException {
			if (value.compareTo(BI_MIN_VALUE)<0 || value.compareTo(BI_MAX_VALUE)>0) throw new IllegalArgumentException("Argument is not within range");		
			this.value = value.compareTo(HALF)<0 ? value.longValue() : value.subtract(HALF).longValue() | 0x80000000l;
		}
	    
	    public Mutable(String stringValue) throws IllegalArgumentException {
	        long value = Long.parseLong(stringValue);
	    	if ( value<L_MIN_VALUE ) throw new IllegalArgumentException("Argument is not within range");
	        this.value = (int) value;
	    }	
	    
	    public static Mutable fromBits(long bits) {
	    	Mutable result = new Mutable();
	    	result.value = bits;
	    	return result;
	    }	    
	    
	    public void setFromBits(int intBits) {
	    	this.value = intBits;
	    }
	    
	    public void setValue(int value) {
	    	if ( value<L_MIN_VALUE ) throw new IllegalArgumentException("Argument is not within range");
	        this.value = value;
	    }
	    
	    public void setValue(long value) {
	    	if ( value<L_MIN_VALUE ) throw new IllegalArgumentException("Argument is not within range");
	        this.value = (int) value;
	    }

	    public void setValue(BigInteger value) {
			if (value.compareTo(BI_MIN_VALUE)<0 || value.compareTo(BI_MAX_VALUE)>0) throw new IllegalArgumentException("Argument is not within range");		
			this.value = value.compareTo(HALF)<0 ? value.longValue() : value.subtract(HALF).longValue() | 0x80000000l;
	    }	    
	    
	}
	
	public final static class Immutable extends UnsignedLong {
		
		private static final long serialVersionUID = 1L;

		Immutable() {}
		
		public Immutable(int value) throws IllegalArgumentException {
	    	if ( value<L_MIN_VALUE ) throw new IllegalArgumentException("Argument is not within range");
	        this.value = value;
	    }

	    public Immutable(long value) throws IllegalArgumentException {
	    	if ( value<L_MIN_VALUE ) throw new IllegalArgumentException("Argument is not within range");
	        this.value = (int) value;
	    }

	    public Immutable(BigInteger value) throws IllegalArgumentException {
			if (value.compareTo(BI_MIN_VALUE)<0 || value.compareTo(BI_MAX_VALUE)>0) throw new IllegalArgumentException("Argument is not within range");		
			this.value = value.compareTo(HALF)<0 ? value.longValue() : value.subtract(HALF).longValue() | 0x80000000l;
		}
	    
	    public Immutable(String stringValue) throws IllegalArgumentException {
	        long value = Long.parseLong(stringValue);
	    	if ( value<L_MIN_VALUE ) throw new IllegalArgumentException("Argument is not within range");
	        this.value = (int) value;
	    }	
	    
	}	
	
    public long toBits() {
    	return value;
    }
	
	@Override
	public int intValue() {
		return (int) value;
	}
	
	@Override
	public long longValue() {
		return value;
	}
	
	@Override
	public float floatValue() {
		return value;
	}
	@Override
	public double doubleValue() {
		return value;
	}
	
	/**
	 * Create big integer value
	 * 
	 * @return new big integer
	 */
	public BigInteger toBigInteger() {
		return value>=0 ? BigInteger.valueOf(value) : BigInteger.valueOf(value & 0x7fffffff).add(HALF); 
	}	
	
    @Override
	public boolean equals(Object obj) {
		if (obj == null) return false;
		if (obj == this) return true;
		
		if (obj instanceof UnsignedLong == false) return false;
		UnsignedLong other = (UnsignedLong) obj;
		return value == other.value;
	}
    
    @Override
    public String toString() {
    	return value>=0 ? Long.toString(value) : toBigInteger().toString();
    }
    
	@Override
	public int compareTo(Number obj) {
		return value>=0 ? Long.signum( value - obj.longValue() ) : 1;
	}
	
	@Override
	public int hashCode() {
	    return (int)value | (int)(value>>32);
	}

	// Initialize Cache
	private static int CACHE_SIZE = 16;
	private static UnsignedLong.Immutable[] CACHE;
	static {
		CACHE = new UnsignedLong.Immutable[CACHE_SIZE];
		for (int i=0; i<CACHE_SIZE; i++) CACHE[i] = new UnsignedLong.Immutable(i);
		ZERO = MIN_VALUE = CACHE[0];
		MAX_VALUE = UnsignedLong.fromBits(0xFFFFFFFF);
	}

}
