/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding.mutable;

import java.util.Set;

import org.simantics.databoard.adapter.AdaptException;
import org.simantics.databoard.adapter.AdapterFactory;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.VariantBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.reflection.ClassBindingFactory;
import org.simantics.databoard.type.Datatype;

/**
 * ImmutableVariantBinding binds VariantType to {@link Variant} Class. 
 */
public class ImmutableVariantBinding extends VariantBinding {
		
	ClassBindingFactory bindingFactory;
	AdapterFactory adapterFactory;
	
	public ImmutableVariantBinding(ClassBindingFactory bindingFactory, AdapterFactory adapterFactory) {
		this.bindingFactory = bindingFactory;
		this.adapterFactory = adapterFactory;
	}
	
	@Override
	public boolean isImmutable() {
		return true;
	}
	
	@Override
	public Object create(Binding binding, Object value) throws BindingException {
		return new Variant(binding, value);
	}
	
	@Override
	public Binding getContentBinding(Object variant) throws BindingException {
		Variant result = (Variant) variant;
		return result.binding;
	}
	
	@Override
	public Datatype getContentType(Object variant) throws BindingException {
		Variant result = (Variant) variant;
		return result.binding.type();
	}
	
	@Override
	public Object getContent(Object variant, Binding binding) throws BindingException {
		Variant var = (Variant) variant;
		try {
			Binding binding1 = var.binding;
			Object value1 = var.value;
			Binding binding2 = binding;
			Object value2 = adapterFactory.adapt(value1, binding1, binding2);
			return value2;
		} catch (AdaptException e) {
			throw new BindingException(e);
		}
	}
	
	public Object getContent(Object variant) throws BindingException {
		Variant var = (Variant) variant;
		return var.value;
	}

	@Override
	public void setContent(Object variant, Binding binding, Object value)
			throws BindingException {
		Variant var = (Variant) variant;
		if (var.binding == binding && var.value==value) return;
		throw new BindingException("Cannot write to immutable variant");
	}

	@Override
	public void assertInstaceIsValid(Object obj, Set<Object> validInstances) throws BindingException {
		if (obj==null) throw new BindingException("null value is not Variant");
		if (obj instanceof Variant == false) throw new BindingException("wrong class, Variant expected");
		Variant var = (Variant) obj;
		if (var.binding==null) throw new BindingException("Binding is expected");		
		var.binding.assertInstaceIsValid(var.value, validInstances);
	}

	@Override
	public boolean isInstance(Object obj) {
		if (obj==null) return false;
		if (obj instanceof Variant == false) return false;
		return true;
	}
	
	boolean isValid(Binding binding, Object obj) throws BindingException {
		binding.assertInstaceIsValid(obj);
		return true;
	}

	@Override
	public int baseHashCode() {
	    return super.baseHashCode() + 7 * adapterFactory.hashCode() + 13 * bindingFactory.hashCode(); 
	}
	
	@Override
	protected boolean baseEquals(Object obj) {
		ImmutableVariantBinding o = (ImmutableVariantBinding)obj;
		return super.baseEquals(obj) && o.adapterFactory == adapterFactory && o.bindingFactory == bindingFactory;
	}
}
