/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding.impl;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import org.simantics.databoard.binding.ArrayBinding;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.MapBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.reflection.VoidBinding;
import org.simantics.databoard.type.MapType;
import org.simantics.databoard.type.RecordType;

/**
 * Binds java.util.TreeSet to Map(T, {})
 *
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
@SuppressWarnings("unchecked")
public class TreeSetBinding extends MapBinding {
	
	public TreeSetBinding(MapType mapType, Binding elementBinding) {
		super(mapType, elementBinding, VoidBinding.VOID_BINDING);
	}
	
	public TreeSetBinding(Binding elementBinding) {
		super(new MapType(elementBinding.type(), RecordType.VOID_TYPE), elementBinding, VoidBinding.VOID_BINDING);
	}

	@Override
	public void clear(Object set) throws BindingException {
		TreeSet<Object> _set = (TreeSet<Object>) set;
		_set.clear();
	}

	@Override
	public boolean containsKey(Object set, Object key) throws BindingException {
		TreeSet<Object> _set = (TreeSet<Object>) set;
		return _set.contains(key);
	}

	@Override
	public boolean containsValue(Object set, Object value)
			throws BindingException {
		return false;
	}

    @Override
    public Object create() throws BindingException {
        return new TreeSet<Object>( getKeyBinding() );
    }

	public Object create(Set<?> initialSet) throws BindingException {
	    if (initialSet instanceof TreeSet && ((TreeSet<?>) initialSet).comparator() == getKeyBinding())
	        return initialSet;
	    
		TreeSet<Object> result = new TreeSet<Object>(getKeyBinding());
		result.addAll(initialSet);
		return result;
	}

	@Override
	public Object create(Map<?, ?> initialMap) throws BindingException {
		TreeSet<Object> result = new TreeSet<Object>( getKeyBinding() );
		result.addAll(initialMap.keySet());
		return result;
	}

	@Override
	public Object create(Object[] keys, Object[] values)
	throws BindingException {		
		TreeSet<Object> result = new TreeSet<Object>( getKeyBinding() );
		for (int i=0; i<keys.length; i++) {
			result.add( keys[i] );
		}
		return result;
	}
	
	@Override
	public Object create(List<Object> keys, List<Object> values) {
		TreeSet<Object> result = new TreeSet<Object>( getKeyBinding() );
		for (int i=0; i<keys.size(); i++)
			result.add(keys.get(i));
		return result;
	}	

	@Override
	public Object get(Object set, Object key) throws BindingException {
		return null;
	}
	
    @Override
    public <K, V> void getAll(Object setFrom, Map<K, V> to) throws BindingException {
		Map<K, V> _to = (Map<K, V>) to;
		TreeSet<K> _setFrom = (TreeSet<K>) setFrom;
		for (K k : _setFrom)
			_to.put(k, null);
	}

	@Override
	public void getAll(Object setFrom, Object[] keys, Object[] values)
			throws BindingException {
		TreeSet<Object> _setFrom = (TreeSet<Object>) setFrom;
		int i=0;
		for (Object k : _setFrom) {
			keys[i] = k;
			values[i] = null;
			i++;
		}
	}

	@Override
	public Object[] getKeys(Object set) throws BindingException {
		TreeSet<Object> _set = (TreeSet<Object>) set;
		return _set.toArray(new Object[_set.size()]);
	}

	@Override
	public void getKeys(Object set, Set<Object> keys) throws BindingException {
		TreeSet<Object> s = (TreeSet<Object>) set;
		keys.addAll(s);
	}
	
	@Override
	public Object[] getValues(Object set) throws BindingException {
		TreeSet<Object> _set = (TreeSet<Object>) set;
		return new Object[_set.size()];
	}
	
	@Override
	public int count(Object src, Object from, boolean fromInclusive, Object end, boolean endInclusive) throws BindingException {
		TreeSet<Object> m = (TreeSet<Object>) src;		
		Set<Object> sm = m.subSet(from, fromInclusive, end, endInclusive);
		return sm.size();
	}
	
	@Override
	public int getEntries(Object src, Object from, boolean fromInclusive, Object end, boolean endInclusive, ArrayBinding dstKeyArrayBinding, Object dstKeyArray, ArrayBinding dstValueArrayBinding, Object dstValueArray, int limit) throws BindingException {
		return 0;
	}

	@Override
	public void put(Object set, Object key, Object value)
			throws BindingException {
		TreeSet<Object> _set = (TreeSet<Object>) set;
		if (value!=null) throw new BindingException("Cannot put non-null to a Set");
		_set.add(key);
	}

	public void putAll(Object setTo, Set<?> from) {
		TreeSet<Object> _set = (TreeSet<Object>) setTo;
		_set.addAll(from);
	}
	
	@Override
	public <K,V> void putAll(Object setTo, Map<K,V> from) throws BindingException {
		TreeSet<Object> _set = (TreeSet<Object>) setTo;
		_set.addAll(from.keySet());
	}

	@Override
	public Object remove(Object set, Object key) throws BindingException {
		TreeSet<Object> _set = (TreeSet<Object>) set;
		_set.remove(key);
		return null;
	}

	@Override
	public int size(Object set) throws BindingException {
		TreeSet<Object> _set = (TreeSet<Object>) set;
		return _set.size();
	}

	@Override
	public boolean isInstance(Object obj) {
		return obj instanceof TreeSet;
	}

	@Override
	public Object getCeilingKey(Object set, Object key) {
		TreeSet<Object> _set = (TreeSet<Object>) set;
		return _set.ceiling(key);
	}

	@Override
	public Object getFirstKey(Object set) {
		TreeSet<Object> _set = (TreeSet<Object>) set;
		return _set.first();
	}

	@Override
	public Object getFloorKey(Object set, Object key) {
		TreeSet<Object> _set = (TreeSet<Object>) set;
		return _set.floor(key);
	}

	@Override
	public Object getHigherKey(Object set, Object key) {
		TreeSet<Object> _set = (TreeSet<Object>) set;
		return _set.higher(key);
	}

	@Override
	public Object getLastKey(Object set) {
		TreeSet<Object> _set = (TreeSet<Object>) set;
		return _set.last();
	}

	@Override
	public Object getLowerKey(Object set, Object key) {
		TreeSet<Object> _set = (TreeSet<Object>) set;
		return _set.lower(key);
	}
	

}

