/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding.impl;

import java.lang.reflect.Array;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.Set;

import org.simantics.databoard.binding.ArrayBinding;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.StringBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.error.UnsupportedOperationException;
import org.simantics.databoard.type.ArrayType;
import org.simantics.databoard.type.StringType;
import org.simantics.databoard.util.IdentityPair;

/**
 * Binds String[] type to String[] class.
 */
public final class StringArrayBinding extends ArrayBinding {
	
	public static StringArrayBinding createFrom(ArrayType type)
	{
		return new StringArrayBinding(type, new StringBindingDefault((StringType) type.componentType));
	}

	public StringArrayBinding(ArrayType type, Binding componentBinding)
	{
		super(type, componentBinding);
	}

	@Override
	public Object create() {
		return new String[0];
	}
	
	@Override
	public Object create(int length, Iterator<Object> values) {
		String[] array = new String[length];
		for(int i=0;i<length;++i)
			array[i] = (String)values.next();
		return array;
	}
	
	@Override
	public void readFrom(Binding srcBinding, Object src, Object dst)
			throws BindingException {
		// Src Binding
		ArrayBinding sb = (ArrayBinding) srcBinding;
		// Src Component Binding
		StringBinding scb = (StringBinding) sb.getComponentBinding();
		
		String[] d = (String[]) dst;
		if (d.length != sb.size(src)) throw new BindingException("String[] is length immutable");
		
		for (int i=0; i<d.length; i++) {
			d[i] = scb.getValue( sb.get(src, i) );
		}
	}	
	
	/**
	 * Create an array object.
	 * Note! The argument is consumed (is used in the result).
	 * 
	 * @param array
	 * @return an object that contains the array
	 */
	public Object create(String[] array) {
		return array.clone();
	}
	
	public Object create(Object[] values) {
		String[] result = new String[values.length];
		for (int i=0; i<values.length; i++)
			result[i] = (String) values[i];
		return result;
	}	
	

	@Override
	public Object get(Object array, int index) throws BindingException {
		if (!isInstance(array)) throw new BindingException("Unexpected class "+array.getClass().getSimpleName()+", String[] expected");
		return ((String[])array)[index];
	}
	
	@Override
	public void getAll(Object array, Object[] result) throws BindingException {
		String[] a = (String[])array;
		for (int i=0; i<a.length; i++)
			result[i] = a[i];
	}
	
	@Override
	public void set(Object array, int index, Object value)
			throws BindingException {
		String[] a = (String[])array;
		a[index] = (String) value;
	}

	@Override
	public int size(Object array) throws BindingException {
		if (!isInstance(array)) throw new BindingException("Unexpected class "+array.getClass().getSimpleName()+", String[] expected");
		return ((String[])array).length;
	}

	@Override
	public boolean isInstance(Object obj) {
		return obj instanceof String[];
	}

	@Override
	public boolean isImmutable() {
		return false;
	}
	
	public String[] getArray(Object array) throws BindingException {
		if (!isInstance(array)) throw new BindingException("Unexpected class "+array.getClass().getSimpleName()+", String[] expected");
		return (String[]) array;
	}	

    @Override
    public int deepHashValue(Object value, IdentityHashMap<Object, Object> hashedObjects) throws BindingException {
    	int result = 1;
    	String[] array = (String[]) value;
    	for (int i=0; i<array.length; i++) {
    		String element = array[i];
    		result = 31*result + element.hashCode();
    	}
    	return result;
    }
    
    @Override
    public int deepCompare(Object o1, Object o2,
    		Set<IdentityPair<Object, Object>> compareHistory)
    		throws BindingException {
    	String[] a1 = (String[]) o1;
    	String[] a2 = (String[]) o2;
		// Compare Lengths
		int l1 = a1.length;
		int l2 = a2.length;
		int dif = l1 - l2;
		if (dif!=0) return dif;
		// Compare elements
		for (int i=0; i<l1; i++) {
			String v1 = a1[i];
			String v2 = a2[i];
			dif = v1.compareTo(v2);
			if (dif!=0) return dif;
		}
		return 0;
    }    

    @Override
    public void add(Object array, int index, Object element)
    		throws BindingException, IndexOutOfBoundsException {
    	throw new UnsupportedOperationException();
    }
    
	@Override
	public void remove(Object array, int index, int count) throws BindingException {
    	throw new UnsupportedOperationException();
	}
	
	@Override
	public void setSize(Object array, int newSize) throws BindingException {
		int oldSize = Array.getLength(array);
		if (oldSize==newSize) return;
		throw new BindingException("String[] is length immutable");
	}	
	
	@Override
	public boolean isResizable() {
		return false;
	}
	
}
