/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding.impl;

import java.util.Set;

import org.simantics.databoard.adapter.AdaptException;
import org.simantics.databoard.adapter.AdapterFactory;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.RecordBinding;
import org.simantics.databoard.binding.VariantBinding;
import org.simantics.databoard.binding.classfactory.TypeClassFactory;
import org.simantics.databoard.binding.error.BindingConstructionException;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.factory.BindingRepository;
import org.simantics.databoard.binding.reflection.BindingRequest;
import org.simantics.databoard.binding.reflection.ClassBindingFactory;
import org.simantics.databoard.type.Datatype;
import org.simantics.databoard.util.Bean;

/**
 * Binds Bean-class (exact match) as Variant.
 *
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public class BeanBinding extends VariantBinding {
	
	ClassBindingFactory bindingFactory;
	AdapterFactory adapterFactory;
	TypeClassFactory typeClassFactory;

	public BeanBinding(ClassBindingFactory bindingFactory, TypeClassFactory typeClassFactory, AdapterFactory adapterFactory) {
		this.bindingFactory = bindingFactory;
		this.adapterFactory = adapterFactory;
		this.typeClassFactory = typeClassFactory;
	}

	public ClassBindingFactory getBindingFactory() {
		return bindingFactory;
	}
	
	public BindingRepository getRepository() {
		return bindingFactory.getRepository();
	}
		
	@Override
	public Object create(Binding srcBinding, Object value) throws BindingException {
//		if (srcBinding instanceof RecordBinding == false) throw new BindingException("Cannot assign "+srcBinding.getClass()+" to bean.");
		try {			
			BindingRequest request = typeClassFactory.getClass( srcBinding.type() );			
			Binding dstBinding = bindingFactory.getBinding( request );
			return adapterFactory.adapt(value, srcBinding, dstBinding); 
		} catch (BindingConstructionException e) {
			throw new BindingException(e);
		} catch (AdaptException e) {
			throw new BindingException(e);
		} 		
	}

	@Override
	public Binding getContentBinding(Object variant) throws BindingException {
		Bean bean = (Bean) variant;
		return bean.getBinding();
	}

	@Override
	public Datatype getContentType(Object variant) throws BindingException {
		Bean bean = (Bean) variant;
		return bean.getBinding().type();
	}

	@Override
	public Object getContent(Object variant, Binding binding)
	throws BindingException {
		try {
			if (binding instanceof RecordBinding == false) throw new BindingException("Cannot assign "+binding.getClass()+" to bean.");
			RecordBinding dstBinding = (RecordBinding) binding;
			Bean src = (Bean) variant;
			RecordBinding srcBinding = (RecordBinding) src.getBinding(); 
			return adapterFactory.adapt(src, srcBinding, dstBinding);
		} catch (AdaptException e) {
			throw new BindingException( e );
		}
	}
	
	@Override
	public Object getContent(Object variant)
	throws BindingException {
		return variant;
	}	

	@Override
	public void setContent(Object variant, Binding binding, Object src)
	throws BindingException {
		if (variant==src) return;
		if (binding instanceof RecordBinding == false) throw new BindingException("Cannot assign "+binding.getClass()+" to bean.");
		RecordBinding srcBinding = (RecordBinding) binding;
		Bean dst = (Bean) variant;
		RecordBinding dstBinding = (RecordBinding) dst.getBinding(); 
		dstBinding.readFrom(srcBinding, src, dst);
	}

	@Override
	public void assertInstaceIsValid(Object obj, Set<Object> validInstances) throws BindingException {
		Bean bean = (Bean) obj;
		bean.getBinding().assertInstaceIsValid(bean, validInstances);
	}

	@Override
	public boolean isInstance(Object obj) {
		return obj instanceof Bean;
	}
	
	@Override
	public boolean isImmutable() {
		return false;
	}

	@Override
	protected boolean baseEquals(Object obj) {
		BeanBinding o = (BeanBinding)obj;
		return super.baseEquals(obj) && o.adapterFactory == this.adapterFactory && o.bindingFactory == this.bindingFactory && o.typeClassFactory == this.typeClassFactory;
	}
	
	@Override
	public int baseHashCode() {
	    return super.baseHashCode() + 7 * adapterFactory.hashCode() + 13 * bindingFactory.hashCode() + 17 * typeClassFactory.hashCode(); 
	}
}
