/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding;

import java.util.IdentityHashMap;
import java.util.Set;
import java.util.regex.Pattern;

import org.simantics.databoard.accessor.reference.ChildReference;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.error.RuntimeBindingException;
import org.simantics.databoard.binding.impl.BindingPrintContext;
import org.simantics.databoard.type.StringType;
import org.simantics.databoard.util.IdentityPair;
import org.simantics.databoard.util.Range;

/**
 * This is a binding of String Type and a Java Object.
 *
 * @see StringType
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public abstract class StringBinding extends Binding {

	public StringBinding(StringType type) {
		this.type = type;
	}
	
	public StringType type() {
		return (StringType) type;
	}
	
	public abstract Object create(String value) throws BindingException;
	
	public Object createUnchecked(String value) throws RuntimeBindingException {
		try {
			return create(value);
		} catch (BindingException e) {
			throw new RuntimeBindingException(e);
		}
	}
	
	@Override
	public void readFrom(Binding srcBinding, Object src, Object dst)
			throws BindingException {
		String v = ((StringBinding)srcBinding).getValue(src);
		setValue(dst, v);
	}
	
	public abstract String getValue(Object o) throws BindingException; 
	public abstract void setValue(Object o, String newValue) throws BindingException; 
	public abstract boolean isInstance(Object obj);
	
    @Override
    public void accept(Visitor1 v, Object obj) {
        v.visit(this, obj);        
    }
    
    @Override
    public <T> T accept(Visitor<T> v) {
        return v.visit(this);
    }

    /**
     * Assert obj is a valid StringType
     * 
     * This asserts that
     *  1. pattern is valid
     *  2. mime type is valid
     *  3. length is valid
     * 
     * @throws BindingException
     */
    @Override
    public void assertInstaceIsValid(Object obj, Set<Object> validInstances) throws BindingException {
		if (!isInstance(obj)) throw new BindingException("Not a correct instance");				
		
    	StringType type = type();
    	Pattern p = type.getCompiledPattern();
    	String mimeType = type.getMimeType();
    	Range length = type.getLength();
    	if (p==null && length==null && mimeType==null) return;    	
    	String str = getValue(obj);    	
    	
    	// 1. Assert the pattern is valid
    	if (p!=null && !p.matcher(str).matches()) {
    		throw new BindingException("The value ("+str+") doesn't match the pattern ("+p.toString()+").");
    	}
    	
    	// 2. Assert the mime type is valid
    	if (mimeType!=null) {    		
//    		try {
//    			MimeType mt = new MimeType(mimeType);
//    			if (!mt.match(str)) 
//    				throw new BindingException("The object is not "+mimeType);
//    		} catch (MimeTypeParseException mtpe) {
//    			throw new BindingException(mtpe);
//    		}    		
    	}
    	
    	// 3. Assert the length is valid
    	if (length!=null && !length.contains(str.length())) {
    		throw new BindingException("Length ("+str.length()+") doesn't fit into the range "+length);
    	}
    }

    @Override
    public int deepCompare(Object o1, Object o2,
    		Set<IdentityPair<Object, Object>> compareHistory)
    		throws BindingException {
    	String v1 = getValue(o1);
    	String v2 = getValue(o2);
    	return v1.compareTo(v2);    	
    }
    
    @Override
    public int deepHashValue(Object value, IdentityHashMap<Object, Object> hashedObjects) throws BindingException {
    	return getValue(value).hashCode();
    }
    
    @Override
	protected void toString(Object value, BindingPrintContext ctx) throws BindingException {
    	ctx.b.append('"');
    	ctx.b.append(getValue(value));
    	ctx.b.append('"');
    }
    
	@Override
	public Binding getComponentBinding(ChildReference path) {
		if (path==null) return this;		
		throw new IllegalArgumentException();
	}	    
	
	@Override
	public int getComponentCount() {
		return 0;
	}
	
	@Override
	public Binding getComponentBinding(int index) {
		throw new IllegalArgumentException();
	}
    
}
