/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding;

import java.util.IdentityHashMap;
import java.util.Set;

import org.simantics.databoard.accessor.reference.ChildReference;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.error.RuntimeBindingException;
import org.simantics.databoard.binding.impl.BindingPrintContext;
import org.simantics.databoard.binding.impl.BooleanBindingDefault;
import org.simantics.databoard.binding.mutable.MutableBooleanBinding;
import org.simantics.databoard.type.BooleanType;
import org.simantics.databoard.util.IdentityPair;

/**
 * This is a binding of Boolean Type and a Java Object.
 *
 * @see BooleanBindingDefault
 * @see MutableBooleanBinding
 * @see BooleanType
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public abstract class BooleanBinding extends Binding {
	
	public BooleanBinding(BooleanType type) {
		this.type = type;
	}
	
	public abstract Object create(boolean value) throws BindingException; 
	public abstract Object create(Boolean value) throws BindingException; 
	public abstract void setValue(Object obj, Boolean newValue) throws BindingException;
	public abstract void setValue(Object obj, boolean newValue)	throws BindingException;
	public abstract Boolean getValue(Object o) throws BindingException; 
	public abstract boolean getValue_(Object o) throws BindingException; 

	public Object createUnchecked(Boolean value) throws RuntimeBindingException {
		try {
			return create(value);
		} catch (BindingException e) {
			throw new RuntimeBindingException(e);
		}
	}
	
	@Override
	public void readFrom(Binding srcBinding, Object src, Object dst)
			throws BindingException {
		boolean v = ((BooleanBinding)srcBinding).getValue_(src);
		setValue(dst, v);
	}	
	
	@Override
	public BooleanType type() {
		return (BooleanType) super.type();
	}
	
	@Override
    public void accept(Visitor1 v, Object obj) {
        v.visit(this, obj);        
    }
    
	@Override
    public <T> T accept(Visitor<T> v) {
        return v.visit(this);
    }
	
    @Override
    public void assertInstaceIsValid(Object obj, Set<Object> validInstances) throws BindingException {
		if (!isInstance(obj)) throw new BindingException("Not a boolean instance");    	
    }
    
    @Override
    public int deepHashValue(Object value, IdentityHashMap<Object, Object> hashedObjects) throws BindingException {
    	boolean b = getValue_(value);
    	return b ? 1231 : 1237;
    }
    
    @Override
    public int deepCompare(Object o1, Object o2,
    		Set<IdentityPair<Object, Object>> compareHistory)
    		throws BindingException {
    	
    	boolean v1 = getValue_(o1);
    	boolean v2 = getValue_(o2);
    	return (v2 == v1 ? 0 : (v1 ? 1 : -1));    	
    }
    
    @Override
	protected void toString(Object value, BindingPrintContext ctx) throws BindingException {
    	ctx.b.append(getValue_(value));
    }

	@Override
	public Binding getComponentBinding(ChildReference path) {
		if (path==null) return this;		
		throw new IllegalArgumentException();
	}	
	
	@Override
	public int getComponentCount() {
		return 0;
	}
	
	@Override
	public Binding getComponentBinding(int index) {
		throw new IllegalArgumentException();
	}
}
