/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.accessor.interestset;

import java.util.Map;

import org.simantics.databoard.annotations.Optional;
import org.simantics.databoard.type.ArrayType;

/**
 * Interest set of an Array Type.  
 * 
 * @see ArrayType
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public class ArrayInterestSet extends InterestSet {
	
	public static final ArrayInterestSet MONITOR_EVERYTHING = new ArrayInterestSet(true, null, true, null, null, null);
	
	// Notifications
	/** Interested in notification of element assignments, add and removal (excludes the value) */
	public boolean notification;
	/** Interested in notifications to elements by index, (excludes the value). Sorted array of indices. */  
	public @Optional int[] notifications;
	
	// Value Assignments
	/** Interested in all notification and value assignments */ 
	public boolean value;
	/** Interested in values by element (includes change). Sorted array of indices */
	public @Optional int[] values;
	
	// Content
	/** Component Interest of the contents of values. Incl. interest in change and value. <code>null</code> if no interest. */ 
	public @Optional InterestSet componentInterest;
	/** Component Interests of the contents of values of specific elements (incl. interest in change and value), <code>null</code> if no interest. */ 
	public @Optional Map<Integer, InterestSet> componentInterests;
	// TODO Make TreeMap

	/**
	 * Create an array interest set
	 * 
	 * @param notification interested in add, remove and change to assignment of all elements 
	 * @param specificNotifications interested in add, remove and change to assignment of all specific elements
	 * @param values interested in add, remove and change to assignment of all elements, incl. new values
	 * @param specificValues interested in add, remove and change to assignment of all specific elements, incl. new values
	 * @param componentInterest Interest of the changes of values and their sub-value changes of all elements
	 * @param componentInterests Interest of the changes of values and their sub-value changes of specific elements
	 */
	public ArrayInterestSet(boolean notification, int[] specificNotifications, boolean values, int[] specificValues, InterestSet componentInterest, Map<Integer, InterestSet> componentInterests) {
		this.notification = notification;
		this.notifications = specificNotifications;
		this.value = values;
		this.values = specificValues;
		this.componentInterest = componentInterest;
		this.componentInterests = componentInterests;
	}

	/**
	 * Returns true, if interested in element assignment, add, removal. 
	 * This excludes interest in the new values.
	 * 
	 * @return true if interested in notifications
	 */
	public boolean inNotifications() {
		return notification | value | (componentInterest!=null);
	}

	/**
	 * Returns true, if interested in assignment of the element at specific <tt>index</tt>
	 * 
	 * @param elementIndex
	 * @return true if interested in notifications of a specific element
	 */
	public boolean inNotificationsOf(int elementIndex) {
		if (notification || value) return true;
		
		if (notifications!=null) {
			for (int i=0; i<notifications.length; i++) {
				if (notifications[i] == elementIndex) return true;
			}
		}
		
		if (values!=null) {
			for (int i=0; i<values.length; i++) {
				if (values[i] == elementIndex) return true;
			}
		}
		
		if (componentInterests!=null) {
			InterestSet compositeInterest = componentInterests.get( elementIndex );
			return compositeInterest!=null;
		}
		return false;
	}
	
	/**
	 * Returns true, if interested in all new value assignments
	 * 
	 * @return true if interested in values of any element
	 */
	public boolean inValues() {
		return value | (componentInterest!=null);
	}

	/**
	 * Returns true, if interested in value assignment of a element at 
	 * <tt>index</tt>.
	 * 
	 * @param elementIndex
	 * @return true if interested in the values of a specific element
	 */
	public boolean inValuesOf(int elementIndex) {
		if (value | componentInterest!=null) return true;
		
		if (values!=null) {
			for (int i=0; i<values.length; i++) {
				if (values[i] == elementIndex) return true;
			}
		}
		
		if (componentInterests!=null?componentInterests.containsKey(elementIndex):false) return true;
		return false;
	}

	/**
	 * Get the interest of the contents of every value 
	 * 
	 * @return component interest set or <code>null</code>
	 */
	public InterestSet getComponentInterest() {
		return componentInterest;
	}
	
	/**
	 * Get composite interest of component value of element at <code>index</code>
	 * 
	 * @param index element index
	 * @return composite interest set or <code>null</code>
	 */
	public InterestSet getComponentInterest(int index) {
		if (componentInterests==null) return null;
		return componentInterests.get(index);		
	}
	
}

