/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.accessor.impl;

import java.util.Collection;
import java.util.Collections;
import java.util.concurrent.Executor;

import org.simantics.databoard.accessor.Accessor.Listener;
import org.simantics.databoard.accessor.event.Event;
import org.simantics.databoard.accessor.interestset.InterestSet;
import org.simantics.databoard.accessor.reference.ChildReference;

/**
 * ListenerEntry is a linked list of <Listener, InterestSet> pairs
 *
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public class ListenerEntry {

	public Listener listener;
	public InterestSet interestSet;
	// Path from the listened object 
	public ChildReference path;
	public Executor executor;
	public ListenerEntry next;
	
	public ListenerEntry(Listener listener, InterestSet interestSet, ChildReference path, Executor executor) {
		if (interestSet==null) throw new IllegalArgumentException();
		this.listener = listener;
		this.interestSet = interestSet;
		this.path = path;
		this.executor = executor;
	}
	
	/**
	 * Add new entry to listener list 
	 * 
	 * @param prevValue
	 * @param listener
	 * @param interestSet
	 * @return the new assignment for the list
	 */
	public static ListenerEntry link(ListenerEntry prevValue, Listener listener, InterestSet interestSet, ChildReference path, Executor executor) {
		ListenerEntry e = new ListenerEntry(listener, interestSet, path, executor);
		e.next = prevValue;
		return e;
	}
	
	@SuppressWarnings("unchecked")
	public <T extends InterestSet> T getInterestSet() {
		return (T) interestSet;
	}

	
	/**
	 * Remove listener entry from linked list
	 * 
	 * @param firstEntry
	 * @param listener the value to be removed
	 * @return the new assignment for the list
	 */
	public static ListenerEntry remove(ListenerEntry firstEntry, Listener listener) {
		ListenerEntry e = firstEntry;
		ListenerEntry p = null;
		while (e!=null) {
			if (e.listener == listener) {
				if (p==null) return e.next;
				p.next = e.next;
				return firstEntry;
			}
			p = e;
			e = e.next;
		}
		return null;
	}
	
	/**
	 * Emits an event to the listener. 
	 * 
	 * NOTE, Event instance can be emited only once. This method writes to the
	 * e.reference field. 
	 *  
	 * @param e
	 */
	public void emitEvent(final Event e) {
//		e.reference = ChildReference.concatenate(path, e.reference);
		if (executor == null) {
			listener.onEvents( Collections.singleton(e) );
		} else {
			executor.execute( new Runnable() {
				public void run() {
					listener.onEvents( Collections.singleton(e) );
				}});
		}
	}

	public void emitEvents(final Collection<Event> events) {
//		for (Event e : events) {
//			e.reference = ChildReference.concatenate(path, e.reference);
//		}
		if (executor == null) {
			listener.onEvents( events );
		} else {
			executor.execute( new Runnable() {
				public void run() {
					listener.onEvents( events );
				}});
		}
	}
	
}

