/*******************************************************************************
 * Copyright (c) 2007, 2012 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.swt.contentassist;

import java.util.Objects;

import org.simantics.browsing.ui.common.modifiers.EnumeratedValue;
import org.simantics.browsing.ui.common.modifiers.Enumeration;
import org.simantics.db.Session;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;

/**
 * @author Tuukka Lehtonen
 */
public abstract class GraphContentAssistEnumerationModifier<T> extends AbstractContentAssistEnumerationModifier<T> {

    protected final Session session;

    public GraphContentAssistEnumerationModifier(Session session, Enumeration<T> enumeration, T value) {
        this(session, enumeration, enumeration.find(value));
    }

    public GraphContentAssistEnumerationModifier(Session session, Enumeration<T> enumeration, EnumeratedValue<T> value) {
        super(enumeration, value);
        if (session == null)
            throw new NullPointerException("null session");
        this.session = session;
    }

    @Override
    protected void modifyWithObject(final T oldEnumObject, final T enumObject, boolean force) {
        if (!force && Objects.equals(oldEnumObject, enumObject))
            return;

        try {
            session.getSession().syncRequest(new WriteRequest() {
                @Override
                public void perform(WriteGraph graph) throws DatabaseException {
//                    System.out.println("old enum object: " + oldEnumObject);
//                    System.out.println("enum object: " + enumObject);
                    modifyWithObject(graph, oldEnumObject, enumObject);
                }
            });
        } catch (DatabaseException e) {
            handleException(e);
        }
    }

    /**
     * Implement this modifier to perform database modifications.
     * 
     * @param graph
     * @param oldEnumObject
     * @param enumObject
     */
    protected abstract void modifyWithObject(WriteGraph graph, T oldEnumObject, T enumObject) throws DatabaseException;

    /**
     * Client overridable handler for possible exceptions that may occur while
     * performing the database modifications.
     * 
     * <p>
     * This default implementation merely forwards the exception as a runtime
     * exception.
     * 
     * @param e
     */
    protected void handleException(DatabaseException e) {
        throw new RuntimeException(e);
    }

}
