/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.swt;

import java.util.HashMap;

import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.swt.widgets.Widget;
import org.simantics.browsing.ui.BuiltinKeys;
import org.simantics.browsing.ui.CheckedState;
import org.simantics.browsing.ui.GraphExplorer;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.PrimitiveQueryUpdater;
import org.simantics.browsing.ui.NodeContext.PrimitiveQueryKey;
import org.simantics.browsing.ui.common.processors.AbstractPrimitiveQueryProcessor;
import org.simantics.browsing.ui.common.processors.IsCheckedProcessor;
import org.simantics.browsing.ui.common.processors.ProcessorLifecycle;

/**
 * @author Tuukka Lehtonen
 */
public class DefaultIsCheckedProcessor extends AbstractPrimitiveQueryProcessor<CheckedState> implements
IsCheckedProcessor, ProcessorLifecycle {

    private static final boolean DEBUG = false;

    /**
     * The map of node contexts that currently have another checked state than
     * {@link CheckedState#NOT_CHECKED}.
     */
    private final HashMap<NodeContext, CheckedState>          checkedStates  = new HashMap<NodeContext, CheckedState>();
    private final HashMap<NodeContext, PrimitiveQueryUpdater> checkedQueries = new HashMap<NodeContext, PrimitiveQueryUpdater>();

    private final CheckedState                                defaultState;

    private Tree                                              tree;

    private boolean                                           viewerSupportsChecking;

    public DefaultIsCheckedProcessor() {
        this(CheckedState.NOT_CHECKED);
    }

    public DefaultIsCheckedProcessor(CheckedState defaultState) {
        if (defaultState == null)
            throw new NullPointerException("null default state");
        this.defaultState = defaultState;
    }

    @Override
    public Object getIdentifier() {
        return BuiltinKeys.IS_CHECKED;
    }

    @Override
    public String toString() {
        return "IsCheckedProcessor";
    }

    @Override
    public CheckedState query(PrimitiveQueryUpdater updater, NodeContext context, PrimitiveQueryKey<CheckedState> key) {
        // Don't gather records if the Tree we are attached to does not support
        // checked items. This optimizes memory consumption while allowing this
        // processor to exist in the attached GraphExplorer.
        if (!viewerSupportsChecking)
            return defaultState;

        CheckedState checked = checkedStates.get(context);

        if (DEBUG)
            System.out.println("isChecked(" + updater + ", " + context + "): " + checked);

        checkedQueries.put(context, updater);
        return checked != null ? checked : CheckedState.NOT_CHECKED;
    }

    @Override
    public boolean setChecked(NodeContext context, CheckedState checked) {
        return _setChecked(context, checked);
    }

    private boolean _setChecked(NodeContext context, CheckedState checked) {
        if (checked != defaultState) {
            return this.checkedStates.put(context, checked) != checked;
        } else {
            return this.checkedStates.remove(context) != null;
        }
    }

    Listener treeListener = new Listener() {
        @Override
        public void handleEvent(Event event) {
            NodeContext context = (NodeContext) event.item.getData();
            switch (event.type) {
                case SWT.Selection:
                    if (event.detail == SWT.CHECK && event.item != null) {
                        TreeItem item = (TreeItem) event.item;
                        boolean checked = item.getChecked();
                        boolean grayed = item.getGrayed();
                        nodeStatusChanged(context, toCheckedState(checked, grayed));
                    }
                    break;
            }
        }
    };

    protected void nodeStatusChanged(NodeContext context, CheckedState checked) {
        if (DEBUG)
            System.out.println("isChecked status changed for " + context + ": " + checked);

        _setChecked(context, checked);
//        PrimitiveQueryUpdater updater = checkedQueries.get(context);
//        if (updater != null)
//            updater.scheduleReplace(context, BuiltinKeys.IS_CHECKED, checked);
    }

    @Override
    public void attached(GraphExplorer explorer) {
        if (DEBUG)
            System.out.println(this + " attaching to " + explorer);

        Object control = explorer.getControl();
        if (control instanceof Tree) {
            this.tree = (Tree) control;
            viewerSupportsChecking = supportsChecking(tree);
            if (viewerSupportsChecking) {
                if (DEBUG)
                    System.out.println("\ttree supports checking, adding listener");
                tree.addListener(SWT.Selection, treeListener);
            }
        } else {
            System.out.println("WARNING: " + getClass().getSimpleName() + " attached to unsupported control: " + control);
        }
    }

    @Override
    public void clear() {
        checkedStates.clear();
        checkedQueries.clear();
    }

    @Override
    public void detached(GraphExplorer explorer) {
        if (DEBUG)
            System.out.println(this + " detaching from " + explorer);

        clear();
        if (tree != null) {
            if (viewerSupportsChecking) {
                if (DEBUG)
                    System.out.println("\ttree supported checking, removing listener");
                tree.removeListener(SWT.Selection, treeListener);
            }
            tree = null;
            viewerSupportsChecking = false;
        }
    }

    private static boolean supportsChecking(Widget control) {
        return (control.getStyle() & SWT.CHECK) != 0;
    }

    private static CheckedState toCheckedState(boolean checked, boolean grayed) {
        if (checked)
            return grayed ? CheckedState.GRAYED : CheckedState.CHECKED;
        return CheckedState.NOT_CHECKED;
    }

}