/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.common.property;

import java.util.Collection;

import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.common.NodeContextUtil;
import org.simantics.utils.datastructures.slice.SliceUtil;
import org.simantics.utils.datastructures.slice.Sliceable;
import org.simantics.utils.datastructures.slice.ValueRange;

/**
 * @author Tuukka Lehtonen
 */
public class PropertyUtil {

    /**
     * Use databoard instead of this?
     */
    public static enum ValueType {
        NoValue,
        BooleanValue,
        ByteValue,
        IntegerValue,
        LongValue,
        FloatValue,
        DoubleValue,
        StringValue,
        UnrecognizedValue;

        public static ValueType convert(Object obj) {
            if (obj == null) return NoValue;
            if (obj.getClass() == Boolean.class) return BooleanValue;
            if (obj.getClass() == boolean[].class) return BooleanValue;
            if (obj.getClass() == Byte.class) return ByteValue;
            if (obj.getClass() == byte[].class) return ByteValue;
            if (obj.getClass() == Integer.class) return IntegerValue;
            if (obj.getClass() == int[].class) return IntegerValue;
            if (obj.getClass() == Long.class) return LongValue;
            if (obj.getClass() == long[].class) return LongValue;
            if (obj.getClass() == Float.class) return FloatValue;
            if (obj.getClass() == float[].class) return FloatValue;
            if (obj.getClass() == Double.class) return DoubleValue;
            if (obj.getClass() == double[].class) return DoubleValue;
            if (obj.getClass() == String.class) return StringValue;
            if (obj.getClass() == String[].class) return StringValue;
            return UnrecognizedValue;
        }

        public static String toString(ValueType type) {
            switch (type) {
                case NoValue: return "no value";
                case BooleanValue: return "boolean";
                case ByteValue: return "byte";
                case DoubleValue: return "double";
                case FloatValue: return "float";
                case IntegerValue: return "int32";
                case LongValue: return "int64";
                case StringValue: return "string";
                case UnrecognizedValue: return "unknown value";
                default:
                    throw new IllegalArgumentException("unrecognized value type: "+ type);
            }
        }
    }


    /**
     * Returns the child nodes of the specified IProperty, constructed with the
     * specified IPropertyFactory.
     * 
     * <p>
     * Child nodes only exist for {@link IArrayProperty} instances whose value
     * range size is > 1.
     * </p>
     * 
     * @param <T>
     * @param property
     * @return child nodes of the specified property
     */
    public static <T extends IProperty> NodeContext[] getChildren(IProperty property) {
        return getChildren(property, false);
    }

    /**
     * Returns the child nodes of the specified IProperty, constructed with the
     * specified IPropertyFactory.
     * 
     * <p>
     * Child nodes only exist for {@link IArrayProperty} instances whose value
     * range size is > 0. You can whether you want to create child nodes for
     * ranges of size 1 or not.
     * </p>
     * 
     * @param <T>
     * @param property
     * @param createOneChild <code>true</code> to create a sub-node even if the
     *        array property range is of size 1
     * @return child nodes of the specified property
     */
    public static <T extends IProperty> NodeContext[] getChildren(IProperty property, boolean createOneChild) {
        // Plain properties do not have children.
        if (!(property instanceof IArrayProperty))
            return NodeContext.NONE;

        IArrayProperty array = (IArrayProperty) property;
        ValueRange range = array.getRange();
        if (range.size() < 1 || (!createOneChild && range.size() < 2))
            return NodeContext.NONE;
        Collection<IArrayProperty> children = subnodify(array, range.start(), range.size());
        return NodeContextUtil.toContextsWithInput(children);
    }

    /**
     * @param property
     * @return {@link Boolean#TRUE} if the property is an {@link IArrayProperty}
     *         and its value range size is > 1.
     */
    public static Boolean hasChildren(IProperty property) {
        if (!(property instanceof IArrayProperty))
            return Boolean.FALSE;

        IArrayProperty array = (IArrayProperty) property;
        ValueRange range = array.getRange();
        return Boolean.valueOf(range.size() < 2);
    }

    /**
     * @param prop
     * @param rangeStart
     * @param rangeSize
     * @return
     */
    public static <T extends Sliceable<T>> Collection<T> subnodify(Sliceable<T> prop, int rangeStart, int rangeSize) {
        return SliceUtil.subnodify(prop, rangeStart, rangeSize);
    }

    /**
     * @param prop
     * @param rangeStart
     * @param rangeSize
     * @param pow10OfItemsPerLevel
     * @return
     */
    public static <T> Collection<T> subnodify(Object p, int rangeStart, int rangeSize, int pow10OfItemsPerLevel) {
        return SliceUtil.subnodify(p, rangeStart, rangeSize, pow10OfItemsPerLevel);
    }

}
