/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.application.arguments;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.TreeMap;

public class Arguments implements IArguments {

    private Map<String, List<IArgument<?>>> args = new TreeMap<>();

    /* (non-Javadoc)
     * @see org.simantics.application.arguments.IArguments#put(org.simantics.application.arguments.IArgument)
     */
    public void put(IArgument<?> arg) {
        List<IArgument<?>> list = args.get(arg.getArgument());
        if(list == null) {
            list = new ArrayList<>();
            args.put(arg.getArgument(), list);
        }
        list.add(arg);
    }

    /* (non-Javadoc)
     * @see org.simantics.application.arguments.IArguments#putAll(org.simantics.application.arguments.IArguments)
     */
    @Override
    public void putAll(IArguments args) {
        for (IArgument<?> arg : args.toArray()) {
            put(arg);
        }
    }

    public List<IArgument<?>> remove(IArgumentFactory<?> key) {
        return (List<IArgument<?>>) args.remove(key.getArgument());
    }

    /* (non-Javadoc)
     * @see org.simantics.application.arguments.IArguments#contains(org.simantics.application.arguments.IArgumentFactory)
     */
    public <T> boolean contains(IArgumentFactory<T> key) {
        return args.containsKey(key.getArgument());
    }

    /* (non-Javadoc)
     * @see org.simantics.application.arguments.IArguments#get(org.simantics.application.arguments.IArgumentFactory)
     */
    public <T> T get(IArgumentFactory<T> key) {
        return get(key, null);
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T> T get(IArgumentFactory<T> key, T defaultValue) {
        List<IArgument<?>> a = (List<IArgument<?>>) args.get(key.getArgument());
        if(a == null)
            return defaultValue;
        if(a.size() == 0)
            return defaultValue;
        return (T)a.iterator().next().getValue();
    }

    /* (non-Javadoc)
     * @see org.simantics.application.arguments.IArguments#getAll(org.simantics.application.arguments.IArgumentFactory)
     */
    @SuppressWarnings("unchecked")
	public <T> List<T> getAll(IArgumentFactory<T> key) {
        List<IArgument<?>> a = (List<IArgument<?>>) args.get(key.getArgument());
        if(a == null)
           return null;
        List<T> result = new ArrayList<>();
        for(IArgument<?> arg : a)
            result.add((T)arg.getValue());
        return result;
    }

    /* (non-Javadoc)
     * @see org.simantics.application.arguments.IArguments#toArray()
     */
    public IArgument<?>[] toArray() {
        ArrayList<IArgument<?>> result = new ArrayList<>();
        for(List<IArgument<?>> l : args.values())
            result.addAll(l);
        return result.toArray(new IArgument<?>[result.size()]);
    }

    /**
     * @param args
     * @param parsedArguments
     * @return
     */
    public static IArguments parse(String[] args, IArgumentFactory<?>... parsedArguments) {
        ListIterator<String> iter = Arrays.asList(args).listIterator();
        IArguments result = new Arguments();
        while (iter.hasNext()) {
            String arg = iter.next();

            IArgument<?> ret = null;
            for (IArgumentFactory<?> factory : parsedArguments) {
                try {
                    if ((ret = factory.parse(arg, iter)) != null)
                        break;
                } catch (IllegalArgumentException e) {
                    // Argument matched, but the argument value was invalid.
                    // TODO: react to this.
                }
            }
            if (ret != null)
                result.put(ret);
        }
        
        return result;
    }

}
