/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.application.arguments;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.simantics.application.internal.Activator;

/**
 * Simple utilities for IApplication implementation.
 */
public class ApplicationUtils {

    /**
     * Appends properties from the specified URL to system properties which are
     * available through {@link System#getProperties()}.
     * 
     * @param props a valid URL or <code>null</code> for no operation.
     */
    public static void loadSystemProperties(URL props) {
        if (props != null) {
            InputStream in = null;
            try {
                in = props.openStream();
                System.getProperties().load(in);
            } catch (IOException e) {
                Activator.getDefault().getLog().log(new Status(IStatus.WARNING, Activator.PLUGIN_ID, "Failed to load system properties: " + props, e));
            } finally {
                try {
                    if (in != null)
                        in.close();
                } catch (IOException e) {
                    Activator.getDefault().getLog().log(new Status(IStatus.WARNING, Activator.PLUGIN_ID, "Failed to close system properties: " + props, e));
                }
            }
        }
    }

    /**
     * Decodes a dot ('.') -separated path into the path elements. The '\'
     * character is used for escaping dots themselves.
     * 
     * <p>Examples of the results:</p>
     * <ul>
     * <li>decodePath("abc") => ["abc"]</li>
     * <li>decodePath("abc.def.efg") => ["abc", "def", "efg"]</li>
     * <li>decodePath("abc\\..def\\..ghi.jkl") => ["abc.def..ghi", "jkl"]</li>
     * <li>decodePath("abc\\..def\\..ghi.jkl\\.") => ["abc.def.ghi", "jkl."]</li>
     * <li>decodePath("abc\\..def\\..ghi.jkl\\..") => ["abc.def.ghi", "jkl."]</li>
     * <li>decodePath("abc\\..def.ghi\\.jkl") => ["abc.def", "ghi.jkl"]</li>
     * <li>decodePath("abc\\..def.ghi\\.jkl.") => ["abc.def", "ghi.jkl"]</li>
     * <li>decodePath("abc\\..def.ghi\\.jkl\\..") => ["abc.def", "ghi.jkl."]</li>
     * </ul>
     * 
     * @param dottedPath
     * @return the elements of the dot-separated path as a String array
     */
    public static String[] decodePath(String dottedPath) {
        if (dottedPath == null)
            return new String[0];

        List<String> result = new ArrayList<String>();

        int index = 0;
        while (dottedPath.length() > 0) {
            int dotIndex = dottedPath.indexOf(".", index);
            // Skip escaped dots
            if (dotIndex < 0) {
                if (dottedPath.length() > 0) {
                    // Take the rest of the string and stop splitting.
                    result.add(dottedPath.replace("\\.", "."));
                }
                break;
            }
            if (dotIndex > 0 && dottedPath.charAt(dotIndex - 1) == '\\') {
                index = dotIndex + 1;
                continue;
            }
            // Grab until the next dot and replace any escaped dots with dots.
            String pathPart = dottedPath.substring(0, dotIndex);
            result.add(pathPart.replace("\\.", "."));
            dottedPath = dottedPath.substring(dotIndex + 1);
            index = 0;
        }

        return result.toArray(new String[result.size()]);
    }
    
    /**
     * Encodes a path of names into a single dot-separated ('.') path string.
     * All '.' characters in the path segment names themselves are escaped with
     * a single '\' character before concatenation into the result.
     * 
     * <p>
     * String encoded with this method can be decoded with
     * {@link #decodePath(String)}.
     * 
     * @param namePath a path of names
     * @return the path encoded into a single string
     */
    public static String encodePath(String[] namePath) {
        StringBuilder b = new StringBuilder();
        boolean first = true;
        for (String s : namePath) {
            if (!first)
                b.append('.');
            first = false;
            b.append(s.replace(".", "\\."));
        }
        return b.toString();
    }

    /**
     * Parse the arguments received by the application (available through
     * {@link Platform#getApplicationArgs()}) for instances of the specified
     * arguments. Found matches are contained in the resulting IArguments
     * instance.
     * 
     * @param accepted the arguments that should be parsed
     * @return the argument matches that were found
     */
    public static IArguments parseApplicationArguments(IArgumentFactory<?>... accepted) {
        return Arguments.parse(Platform.getApplicationArgs(), accepted);
    }
    
//    public static void main(String[] args) {
//        System.out.println(Arrays.toString(decodePath("abc")));
//        System.out.println(Arrays.toString(decodePath("abc.def.efg")));
//        System.out.println(Arrays.toString(decodePath("abc\\..def\\..ghi.jkl")));
//        System.out.println(Arrays.toString(decodePath("abc\\..def\\..ghi.jkl\\.")));
//        System.out.println(Arrays.toString(decodePath("abc\\..def\\..ghi.jkl\\..")));
//        System.out.println(Arrays.toString(decodePath("abc\\..def.ghi\\.jkl")));
//        System.out.println(Arrays.toString(decodePath("abc\\..def.ghi\\.jkl.")));
//        System.out.println(Arrays.toString(decodePath("abc\\..def.ghi\\.jkl\\..")));
//    }

}
