package org.simantics.acorn;

import java.io.ByteArrayInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.Serializable;
import java.nio.file.NoSuchFileException;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.binding.mutable.MutableVariant;
import org.simantics.databoard.util.binary.BinaryMemory;
import org.simantics.db.IO;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MainState implements Serializable {

	private static final long serialVersionUID = 6237383147637270225L;

	private static final Logger LOGGER = LoggerFactory.getLogger(MainState.class);

	public static final String MAIN_STATE = "main.state";

	public int headDir;

	public MainState() {
		this.headDir = 0;
	}

	private MainState(int headDir) {
		this.headDir = headDir;
	}

	public boolean isInitial() {
		return this.headDir == 0;
	}

	public static MainState load(Store store, Runnable rollbackCallback) throws IOException {

		store.ensureExists();
		AcornKey mainState = store.rootKey(MAIN_STATE);

		try {

			byte[] mainStateValue = mainState.bytes();
			ByteArrayInputStream bais = new ByteArrayInputStream(mainStateValue);
			MainState state = (MainState) org.simantics.databoard.Files.readFile(bais, Bindings.getBindingUnchecked(MainState.class));
			bais.close();

			int latestRevision = state.headDir - 1;
			try {
				AcornKey latest = store.rootKey(Integer.toString(latestRevision));
				if (HeadState.validateHeadStateIntegrity(latest.child(HeadState.HEAD_STATE))) {
					archiveRevisionDirectories(store, latestRevision, rollbackCallback);
					return state;
				}
				LOGGER.warn("Failed to start database from revision " + latestRevision + " stored in " + mainState + ". " + HeadState.HEAD_STATE + " is invalid.");
				return rollback(store, rollbackCallback);
			} catch (FileNotFoundException e) {
				LOGGER.warn("Failed to start database from revision " + latestRevision + " stored in " + mainState + ". Revision does not contain " + HeadState.HEAD_STATE + ".");
				return rollback(store, rollbackCallback);
			}
		} catch (IOException e) {
			// The database may also be totally empty at this point
			if (listRevisionDirs(store, true, MainState::isInteger).isEmpty())
				return new MainState(0);

			LOGGER.warn("Unclean exit detected, " + mainState + " not found. Initiating automatic rollback.");
			return rollback(store, rollbackCallback);
		} catch (Exception e) {
			LOGGER.warn("Unclean exit detected. Initiating automatic rollback.", e);
			return rollback(store, rollbackCallback);
		} finally {
			mainState.deleteIfExists();
		}
	}

	private static MainState rollback(Store directory, Runnable rollbackCallback) throws IOException {
		LOGGER.warn("Database rollback initiated for " + directory);
		rollbackCallback.run();
		AcornKey latest = findNewHeadStateDir(directory);
		int latestRevision = latest != null ? safeParseInt(-1, latest) : -1;
		// +1 because we want to return the next head version to use,
		// not the latest existing version.
		MainState state = new MainState( latestRevision + 1 );
		archiveRevisionDirectories(directory, latestRevision, rollbackCallback);
		LOGGER.warn("Database rollback completed. Restarting database from revision " + latest);
		return state;
	}

	private byte[] toByteArray() throws IOException {
		try (BinaryMemory rf = new BinaryMemory(4096)) {
			Bindings.getSerializerUnchecked(Bindings.VARIANT).serialize(rf, MutableVariant.ofInstance(this));
			return rf.toByteBuffer().array();
		}
	}

	public void save(Store store) throws IOException {
		byte[] bytes = toByteArray();
		IO io = store.rootKey(MAIN_STATE).getIO();
		io.saveBytes(bytes, bytes.length, true);
	}

	private static int safeParseInt(int defaultValue, AcornKey p) {
		try {
			return Integer.parseInt(p.getName());
		} catch (NumberFormatException e) {
			return defaultValue;
		}
	}

	private static boolean isInteger(AcornKey p) {
		return safeParseInt(Integer.MIN_VALUE, p) != Integer.MIN_VALUE;
	}

	private static Predicate<AcornKey> isInteger() {
		return p -> isInteger(p);
	}


	private static Predicate<AcornKey> isGreaterThan(int i) {
		return p -> {
			int pi = safeParseInt(Integer.MIN_VALUE, p);
			return pi != Integer.MIN_VALUE && pi > i;
		};
	}

	/**
	 *  
	 * @param directory
	 * @param callback 
	 * @return
	 * @throws IOException
	 */
	private static AcornKey findNewHeadStateDir(Store store) throws IOException {
		List<AcornKey> dirs = listRevisionDirs(store, true, isInteger()); 
		for (AcornKey last : dirs)
			if (HeadState.validateHeadStateIntegrity(last.child(HeadState.HEAD_STATE)))
				return last;
		return null;
	}

	private static void archiveRevisionDirectories(Store store, int greaterThanRevision, Runnable rollbackCallback) throws IOException {
		List<AcornKey> reverseSortedPaths = listRevisionDirs(store, true, isGreaterThan(greaterThanRevision));
		if (reverseSortedPaths.isEmpty())
			return;

		// If none of the revisions to be archived are actually committed revisions
		// then just delete them. Commitment is indicated by the head.state file.
		if (!anyContainsHeadState(reverseSortedPaths)) {
			for (AcornKey p : reverseSortedPaths) {
				deleteAll(p);
				LOGGER.info("Removed useless working folder " + p);
			}
			return;
		}

		// Some kind of rollback is being performed. There is a possibility that
		// indexes and virtual graphs are out of sync with the persistent database.
		rollbackCallback.run();

		AcornKey recoveryFolder = getRecoveryFolder(store);
		recoveryFolder.ensureExists();
		LOGGER.info("Created new database recovery folder " + recoveryFolder);
		for (AcornKey p : reverseSortedPaths) {
			AcornKey to =recoveryFolder.child(p.getName());
			p.copyTo(to);
			LOGGER.info("Archived revision " + p + " in recovery folder " + recoveryFolder);
		}
	}

	private static boolean anyContainsHeadState(List<AcornKey> paths) throws IOException {
		for (AcornKey p : paths)
			if (p.child(HeadState.HEAD_STATE).exists())
				return true;
		return false;
	}

	private static void deleteAll(AcornKey dir) throws IOException {
		dir.deleteAll();
	}

	private static final DateTimeFormatter RECOVERY_DIR_FORMAT = DateTimeFormatter.ofPattern("yyyy-M-d_HH-mm-ss");

	private static AcornKey getRecoveryFolder(Store store) throws IOException {
		return findNonexistentDir(
				store.rootKey("recovery"),
				RECOVERY_DIR_FORMAT.format(ZonedDateTime.now()));
	}

	private static AcornKey findNonexistentDir(AcornKey inDirectory, String prefix) throws IOException {
		for (int i = 0;; ++i) {
			AcornKey dir = inDirectory.child(i == 0 ? prefix : prefix + "-" + i);
			if (!dir.exists())
				return dir;
		}
	}

	@SafeVarargs
	private static final List<AcornKey> listRevisionDirs(Store store, boolean descending, Predicate<AcornKey>... filters) throws IOException {
		int coef = descending ? -1 : 1;
		try (Stream<AcornKey> dirs = store.directories()) {
			Stream<AcornKey> fs = dirs;
			for (Predicate<AcornKey> p : filters)
				fs = fs.filter(p);
			return fs
					.sorted((p1, p2) -> coef * Integer.compare(Integer.parseInt(p1.getName()),
							Integer.parseInt(p2.getName())))
					.collect(Collectors.toList());
		}
	}

}
