/*******************************************************************************
 * This library is free software licensed under LGPL version 2.1
 * Based on GNU Trove, Copyright (c) 2001, Eric D. Friedman.
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - improvements
 *******************************************************************************/
package gnu.trove.ext;

public class DualIdContainerHashMap<T extends DualIdContainer> extends HashBase<T> {

	public DualIdContainerHashMap(Class<T> clazz) throws Exception {
		super(clazz, clazz.getDeclaredConstructor(int.class,int.class).newInstance(-1, -1));
	}

	/**
	 * Locates the index at which <tt>obj</tt> can be inserted.  if
	 * there is already a value equal()ing <tt>obj</tt> in the set,
	 * returns that value's index as <tt>-index - 1</tt>.
	 *
	 * @param obj an <code>Object</code> value
	 * @return the index of a FREE slot at which obj can be inserted
	 * or, if obj is already stored in the hash, the negative value of
	 * that index, minus 1: -index -1.
	 */
	private final int insertionIndex(final long id) {

		final T[] set = _set;
		final int length = set.length;
		final int hash = (31 * ((int)(id>>>32)) + (int)id) & 0x7fffffff;
		int index = hash % length;
		T cur = set[index];

		if (cur == null) {
			return index;       // empty, all done
		} else if (cur != REMOVED && (id == cur.longId())) {
			return -index -1;   // already stored
		} else {                // already FULL or REMOVED, must probe
			// compute the double hash
			final int probe = 1 + (hash % (length - 2));

			// if the slot we landed on is FULL (but not removed), probe
			// until we find an empty slot, a REMOVED slot, or an element
			// equal to the one we are trying to insert.
			// finding an empty slot means that the value is not present
			// and that we should use that slot as the insertion point;
			// finding a REMOVED slot means that we need to keep searching,
			// however we want to remember the offset of that REMOVED slot
			// so we can reuse it in case a "new" insertion (i.e. not an update)
			// is possible.
			// finding a matching value means that we've found that our desired
			// key is already in the table
			if (cur != REMOVED) {
				// starting at the natural offset, probe until we find an
				// offset that isn't full.
				do {
					index -= probe;
					if (index < 0) {
						index += length;
					}
					cur = set[index];
				} while (cur != null
						&& cur != REMOVED
						&& ! (id == cur.longId()));
			}

			// if the index we found was removed: continue probing until we
			// locate a free location or an element which equal()s the
			// one we have.
			if (cur == REMOVED) {
				int firstRemoved = index;
				while (cur != null
						&& (cur == REMOVED || ! (id == cur.longId()))) {
					index -= probe;
					if (index < 0) {
						index += length;
					}
					cur = set[index];
				}
				// NOTE: cur cannot == REMOVED in this block
						return (cur != null) ? -index -1 : firstRemoved;
			}
			// if it's full, the key is already stored
			// NOTE: cur cannot equal REMOVE here (would have retuned already (see above)
			return (cur != null) ? -index -1 : index;
		}
	}

	private final int removalIndex(final long id) {

		final T[] set = _set;
		final int length = set.length;
		final int hash = ((31 * ((int)(id>>>32)) + (int)id) )& 0x7fffffff;
		int index = hash % length;
		T cur = set[index];

		if ( cur == null ) return -1;

		// NOTE: here it has to be REMOVED or FULL (some user-given value)
		if ( cur == REMOVED || !(id == cur.longId())) {
			// see Knuth, p. 529
			final int probe = 1 + (hash % (length - 2));

			do {
				index -= probe;
				if (index < 0) {
					index += length;
				}
				cur = set[index];
			} while (cur != null
					&& (cur == REMOVED || !(id == cur.longId())));
		}

		return cur == null ? -1 : index;

	}

	/**
	 * Deletes a key/value pair from the map.
	 *
	 * @param key an <code>Object</code> value
	 * @return an <code>Object</code> value
	 */
	public T remove(final long id) {
		T prev = null;
		int index = removalIndex(id);
		if (index >= 0) {
			prev = _set[index];
			removeAt(index);    // clear key,state; adjust size
		}
		return prev;
	}

	/**
	 * Inserts a key/value pair into the map.
	 *
	 * @param key an <code>Object</code> value
	 * @param value an <code>Object</code> value
	 * @return the previous value associated with <tt>key</tt>,
	 * or null if none was found.
	 */
	public final T put(final long id, final T value) {
		T previous = null;
		Object oldValue;
		int index = insertionIndex(id);
		boolean isNewMapping = true;
		if (index < 0) {
			index = -index -1;
			previous = _set[index];
			isNewMapping = false;
			throw new Error("Duplicate entry: " + value);
		}
		oldValue = _set[index];
		_set[index] = value;
		if (isNewMapping) {
			postInsertHook(oldValue == null);
		}
		return previous;
	}

	/**
	 * retrieves the value for <tt>key</tt>
	 *
	 * @param key an <code>Object</code> value
	 * @return the value of <tt>key</tt> or null if no such mapping exists.
	 */
	public final T get(final long id) {

		final T[] set = _set;
		final int length = set.length;
		final int hash = ((31 * ((int)(id>>>32)) + (int)id) )& 0x7fffffff;
		int index = hash % length;
		T cur = set[index];

		if ( cur == null )
			return null;

		// NOTE: here it has to be REMOVED or FULL (some user-given value)
		if ( cur == REMOVED || !(id == cur.longId())) {
			// see Knuth, p. 529
			final int probe = 1 + (hash % (length - 2));

			do {
				index -= probe;
				if (index < 0) {
					index += length;
				}
				cur = set[index];
			} while (cur != null
					&& (cur == REMOVED || !(id == cur.longId())));
		}

		return cur;

	}

	private final int rehashInsertionIndex(final long id, final T[] set) {

		final int length = set.length;
		final int hash = ((31 * ((int)(id>>>32)) + (int)id) )& 0x7fffffff;
		int index = hash % length;
		T cur = set[index];

		if (cur == null) {
			return index;       // empty, all done
		} else if (cur != REMOVED && (id == cur.longId())) {
			return -index -1;   // already stored
		} else {                // already FULL or REMOVED, must probe
			// compute the double hash
			final int probe = 1 + (hash % (length - 2));

			// if the slot we landed on is FULL (but not removed), probe
			// until we find an empty slot, a REMOVED slot, or an element
			// equal to the one we are trying to insert.
			// finding an empty slot means that the value is not present
			// and that we should use that slot as the insertion point;
			// finding a REMOVED slot means that we need to keep searching,
			// however we want to remember the offset of that REMOVED slot
			// so we can reuse it in case a "new" insertion (i.e. not an update)
			// is possible.
			// finding a matching value means that we've found that our desired
			// key is already in the table
			if (cur != REMOVED) {
				// starting at the natural offset, probe until we find an
				// offset that isn't full.
				do {
					index -= probe;
					if (index < 0) {
						index += length;
					}
					cur = set[index];
				} while (cur != null
						&& cur != REMOVED
						&& ! (id == cur.longId()));
			}

			// if the index we found was removed: continue probing until we
			// locate a free location or an element which equal()s the
			// one we have.
			if (cur == REMOVED) {
				int firstRemoved = index;
				while (cur != null
						&& (cur == REMOVED || ! (id == cur.longId()))) {
					index -= probe;
					if (index < 0) {
						index += length;
					}
					cur = set[index];
				}
				// NOTE: cur cannot == REMOVED in this block
						return (cur != null) ? -index -1 : firstRemoved;
			}
			// if it's full, the key is already stored
			// NOTE: cur cannot equal REMOVE here (would have retuned already (see above)
			return (cur != null) ? -index -1 : index;
		}
	}

	/**
	 * rehashes the map to the new capacity.
	 *
	 * @param newCapacity an <code>int</code> value
	 */
	protected final void rehash(int newCapacity) {

		int oldCapacity = _set.length;
		T oldSet[] = _set;
		T newSet[]  = createArray(newCapacity);

		for (int i = oldCapacity; i-- > 0;) {
			if(oldSet[i] != null && oldSet[i] != REMOVED) {
				T o = oldSet[i];
				int index = rehashInsertionIndex(o.longId(), newSet);
				if (index < 0) {
					throwObjectContractViolation(newSet[(-index -1)], o);
				}
				newSet[index] = o;
			}
		}

		_set = newSet;

	}

}
