/*
 * Decompiled with CFR 0.152.
 */
package com.impossibl.postgres.jdbc;

import com.impossibl.postgres.jdbc.ErrorUtils;
import com.impossibl.postgres.jdbc.JDBCSettings;
import com.impossibl.postgres.jdbc.PGDirectConnection;
import com.impossibl.postgres.jdbc.ThreadedHousekeeper;
import com.impossibl.postgres.system.Settings;
import com.impossibl.postgres.system.SystemSettings;
import com.impossibl.postgres.types.SharedRegistry;
import com.impossibl.postgres.utils.guava.Strings;
import io.netty.channel.unix.DomainSocketAddress;
import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.Inet6Address;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

class ConnectionUtil {
    private static final String POSTGRES_UNIX_SOCKET_BASE_NAME = ".s.PGSQL";
    private static final String POSTGRES_UNIX_SOCKET_INVALID_EXT = ".lock";
    private static Logger logger = Logger.getLogger(ConnectionUtil.class.getName());
    private static final Pattern URL_PATTERN = Pattern.compile("jdbc:pgsql:(?://(?:(?<username>[^:@]*)(?::(?<password>[^@]*))?@)?(?<addresses>(?:[a-zA-Z0-9\\-.]+|\\[[0-9a-f:]+])(?::(?:\\d+))?(?:,(?:[a-zA-Z0-9\\-.]+|\\[[0-9a-f:]+])(?::(?:\\d+))?)*)/)?(?<database>[^?&/]+)(?:[?&](?<parameters>.*))?");
    private static final Pattern ADDRESS_PATTERN = Pattern.compile("(?:([a-zA-Z0-9\\-.]+|\\[[0-9a-f:]+])(?::(\\d+))?)");

    ConnectionUtil() {
    }

    static PGDirectConnection createConnection(String url, Properties info, SharedRegistry.Factory sharedRegistryFactory) throws SQLException {
        ConnectionSpecifier connSpec = ConnectionUtil.parseURL(url);
        if (connSpec == null) {
            return null;
        }
        Settings settings = ConnectionUtil.buildSettings(connSpec, info);
        return ConnectionUtil.createConnection(connSpec.addresses, settings, sharedRegistryFactory);
    }

    static PGDirectConnection createConnection(List<SocketAddress> addresses, Settings settings, SharedRegistry.Factory sharedRegistryFactory) throws SQLException {
        SQLException lastException = null;
        ThreadedHousekeeper.Ref housekeeper = null;
        if (settings.enabled(JDBCSettings.HOUSEKEEPER)) {
            housekeeper = ThreadedHousekeeper.acquire();
        }
        for (SocketAddress address : addresses) {
            DomainSocketAddress domainAddress;
            File socketFile;
            if (address instanceof InetSocketAddress) {
                InetSocketAddress inetAddress = (InetSocketAddress)address;
                if (inetAddress.isUnresolved()) {
                    lastException = new SQLException("Connection Error: address '" + inetAddress.getHostString() + "' is unresolved", "8001");
                    continue;
                }
            } else if (address instanceof DomainSocketAddress && !(socketFile = new File((domainAddress = (DomainSocketAddress)address).path())).exists()) {
                lastException = new SQLException("Connection Error: unix socket '" + socketFile + "' does not exist", "8001");
                continue;
            }
            try {
                PGDirectConnection conn = new PGDirectConnection(address, settings, housekeeper);
                conn.init(sharedRegistryFactory);
                return conn;
            }
            catch (IOException e) {
                lastException = ErrorUtils.makeSQLException("Connection Error: ", "8001", e);
            }
        }
        if (lastException == null) {
            lastException = new SQLException("Connection Error: unknown");
        }
        throw lastException;
    }

    private static Settings buildSettings(ConnectionSpecifier connSpec, Properties connectInfo) {
        Settings settings = new Settings(JDBCSettings.JDBC, SystemSettings.SYS, SystemSettings.PROTO);
        settings.setAll(connSpec.getParameters());
        settings.setAll(connectInfo);
        settings.set(SystemSettings.DATABASE_NAME, connSpec.getDatabase());
        settings.set(SystemSettings.DATABASE_URL, connSpec.getURL());
        return settings;
    }

    static ConnectionSpecifier parseURL(String url) {
        try {
            String unixSocketPath;
            String params;
            Matcher urlMatcher = URL_PATTERN.matcher(url);
            if (!urlMatcher.matches()) {
                return null;
            }
            ConnectionSpecifier spec = new ConnectionSpecifier();
            String hosts = Strings.nullToEmpty(urlMatcher.group("addresses"));
            Matcher hostsMatcher = ADDRESS_PATTERN.matcher(hosts);
            while (hostsMatcher.find()) {
                String name2 = hostsMatcher.group(1);
                String port = hostsMatcher.group(2);
                if (port == null || port.isEmpty()) {
                    port = "5432";
                }
                String[] address = new InetSocketAddress(name2, Integer.parseInt(port));
                spec.appendAddress((SocketAddress)address);
            }
            spec.setDatabase(URLDecoder.decode(urlMatcher.group("database"), "UTF-8"));
            String username = urlMatcher.group("username");
            if (username != null) {
                username = URLDecoder.decode(username, "UTF-8");
                spec.addParameter(SystemSettings.CREDENTIALS_USERNAME.getName(), username);
                String password = urlMatcher.group("password");
                if (password != null) {
                    password = URLDecoder.decode(password, "UTF-8");
                    spec.addParameter(SystemSettings.CREDENTIALS_PASSWORD.getName(), password);
                }
            }
            if ((params = urlMatcher.group("parameters")) != null && !params.isEmpty()) {
                for (String nameValue : params.split("&")) {
                    String name3;
                    String[] items = nameValue.split("=");
                    if (items.length == 1) {
                        name3 = URLDecoder.decode(items[0], "UTF-8");
                        spec.addParameter(name3, "");
                        continue;
                    }
                    if (items.length != 2) continue;
                    name3 = URLDecoder.decode(items[0], "UTF-8");
                    String value = URLDecoder.decode(items[1], "UTF-8");
                    spec.addParameter(name3, value);
                }
            }
            if ((unixSocketPath = spec.parameters.getProperty("unixsocket")) != null) {
                spec.parameters.remove("unixsocket");
                File unixSocketFile = new File(unixSocketPath);
                String[] files = unixSocketFile.list((dir, name) -> name.startsWith(POSTGRES_UNIX_SOCKET_BASE_NAME) && !name.endsWith(POSTGRES_UNIX_SOCKET_INVALID_EXT));
                if (files != null && files.length != 0) {
                    if (files.length != 1) {
                        logger.warning("Multiple PostgreSQL unix sockets found in " + unixSocketPath + ", chose " + files[0] + " at random. Specify socket file to remove this warning.");
                    }
                    unixSocketFile = new File(unixSocketFile, files[0]);
                }
                spec.prependAddress((SocketAddress)new DomainSocketAddress(unixSocketFile));
            }
            if (spec.addresses.isEmpty()) {
                spec.appendAddress(new InetSocketAddress("localhost", 5432));
            }
            logger.fine("parseURL: " + url + " => " + spec);
            return spec;
        }
        catch (Throwable e) {
            return null;
        }
    }

    static class ConnectionSpecifier {
        private List<SocketAddress> addresses = new ArrayList<SocketAddress>();
        private String database = null;
        private Properties parameters = new Properties();

        ConnectionSpecifier() {
        }

        String getDatabase() {
            return this.database;
        }

        void setDatabase(String v) {
            this.database = v;
        }

        List<SocketAddress> getAddresses() {
            return this.addresses;
        }

        void prependAddress(SocketAddress v) {
            this.addresses.add(0, v);
        }

        void appendAddress(SocketAddress v) {
            this.addresses.add(v);
        }

        Properties getParameters() {
            return this.parameters;
        }

        void addParameter(String key, String value) {
            this.parameters.setProperty(key, value);
        }

        private <T extends SocketAddress> List<T> getAddresses(Class<T> type) {
            ArrayList<SocketAddress> found = new ArrayList<SocketAddress>();
            for (SocketAddress address : this.addresses) {
                if (!type.isInstance(address)) continue;
                found.add((SocketAddress)type.cast(address));
            }
            return found;
        }

        private String getInetHosts() {
            List<InetSocketAddress> inetAddresses = this.getAddresses(InetSocketAddress.class);
            if (inetAddresses.isEmpty()) {
                return null;
            }
            StringBuilder hosts = new StringBuilder();
            Iterator<InetSocketAddress> addrIter = inetAddresses.iterator();
            while (addrIter.hasNext()) {
                InetSocketAddress addr = addrIter.next();
                if (addr.getAddress() instanceof Inet6Address) {
                    hosts.append("[").append(addr.getHostString()).append("]");
                } else {
                    hosts.append(addr.getHostString());
                }
                if (addr.getPort() != 5432) {
                    hosts.append(':');
                    hosts.append(addr.getPort());
                }
                if (!addrIter.hasNext()) continue;
                hosts.append(",");
            }
            return hosts.toString();
        }

        private String getUnixPath() {
            List<DomainSocketAddress> unixSockets = this.getAddresses(DomainSocketAddress.class);
            if (unixSockets.isEmpty()) {
                return null;
            }
            return unixSockets.get(0).path();
        }

        private SortedSet<Map.Entry<Object, Object>> getSortedParameters() {
            TreeSet<Map.Entry<Object, Object>> sortedParams = new TreeSet<Map.Entry<Object, Object>>(Comparator.comparing(a -> a.getKey().toString()).reversed());
            sortedParams.addAll(this.parameters.entrySet());
            return sortedParams;
        }

        private String getParameterQuery() {
            if (this.parameters.isEmpty()) {
                return null;
            }
            StringBuilder query = new StringBuilder();
            Iterator entryIter = this.getSortedParameters().iterator();
            while (entryIter.hasNext()) {
                Map.Entry entry = (Map.Entry)entryIter.next();
                String key = entry.getKey().toString();
                String value = Strings.emptyToNull(entry.getValue() != null ? entry.getValue().toString() : null);
                try {
                    query.append(URLEncoder.encode(key, "UTF-8"));
                }
                catch (UnsupportedEncodingException e) {
                    query.append(entry.getKey());
                }
                if (value != null) {
                    query.append("=");
                    try {
                        query.append(URLEncoder.encode(value, "UTF-8"));
                    }
                    catch (UnsupportedEncodingException e) {
                        query.append(entry.getValue());
                    }
                }
                if (!entryIter.hasNext()) continue;
                query.append("&");
            }
            return query.toString();
        }

        String getURL() {
            String url = "jdbc:pgsql:";
            String inetHosts = this.getInetHosts();
            if (inetHosts != null) {
                url = url + "//" + inetHosts + "/";
            }
            if (this.database != null) {
                try {
                    url = url + URLEncoder.encode(this.database, "UTF-8");
                }
                catch (UnsupportedEncodingException e) {
                    url = url + this.database;
                }
            }
            String unixPath = this.getUnixPath();
            String query = this.getParameterQuery();
            if (unixPath != null || query != null) {
                url = url + "?";
            }
            if (unixPath != null) {
                url = url + "unixsocket=" + unixPath;
                if (query != null) {
                    url = url + "&";
                }
            }
            if (query != null) {
                url = url + query;
            }
            return url;
        }

        public String toString() {
            return "ConnectionSpecifier[addresses=" + this.addresses + ",database=" + this.getDatabase() + ",parameters=" + this.getParameters() + "]";
        }
    }
}

