package org.simantics.scl.compiler.source.repository;

import java.io.IOException;
import java.util.Collection;
import java.util.Collections;

import org.simantics.scl.compiler.module.repository.ModuleRepository;
import org.simantics.scl.compiler.module.repository.UpdateListener;
import org.simantics.scl.compiler.source.ModuleSource;
import org.simantics.scl.compiler.source.TextualModuleSource;

import gnu.trove.procedure.TObjectProcedure;

/**
 * An interface for locating modules descriptors and listening if they change.
 * An instance of this interface is used to create a {@link ModuleRepository}.
 * 
 * @author Hannu Niemist&ouml;
 */
public interface ModuleSourceRepository {
    /**
     * Returns all module names governed by this repository. Some implementations
     * may return empty collection, even if the contain modules, if it is hard
     * to discover all modules (for example file system based module repository 
     * works like this).
     */
    Collection<String> getModuleNames();
    
    /**
     * Calls the given procedure with all module names returned by {@link #getModuleNames}
     */
    default void forAllModules(TObjectProcedure<String> procedure) {
        for(String module : getModuleNames())
            if(!procedure.execute(module))
                return;
    }
    
    /**
     * Returns the module source of the given module name or null, if the module does not exists.
     * If {@code listener} is not null, it is called when the module contents change.
     */
    ModuleSource getModuleSource(String moduleName, UpdateListener listener);
    
    /**
     * Returns all documentation names governed by this repository. Some implementations
     * may return empty collection, even if the contain documentation.
     */
    default Collection<String> getDocumentationNames() {
        return Collections.emptyList();
    }
    
    /**
     * Calls the given procedure with all documentation names eturned by {@link #getDocumentationNames}
     */
    default void forAllDocumentations(TObjectProcedure<String> procedure) {
        for(String module : getDocumentationNames())
            if(!procedure.execute(module))
                return;
    }

    /**
     * Returns original markdown text for the given documentation name or null
     * if the documentation does not exist.
     */
    default String getDocumentation(String documentationName) {
        return null;
    }
    
    /**
     * Triggers repository to check if module contents have been changed. Some repositories listen
     * changes and report them to update listeners even without manual triggering.
     */
    default void checkUpdates() {
    }

    /**
     * Resets the repository and removes listeners. This is only used during regression testing and shouldn't be called during normal operation.
     */
    default void clear() {
    }
    
    default String getPossibleSourceText(String moduleName) throws IOException {
        ModuleSource moduleSource = getModuleSource(moduleName, null);
        if(moduleSource == null)
            return null;
        if(!(moduleSource instanceof TextualModuleSource))
            return null;
        return ((TextualModuleSource)moduleSource).getSourceText(null);
    }
}
