/*
 * Copyright 2020 Adrien Grand and the lz4-java contributors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Modified by Tuukka Lehtonen - Semantum Oy
 *  - allow extracting native library to designated directory
 */
package net.jpountz.util;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.Arrays;

/** FOR INTERNAL USE ONLY */
public enum Native {
  ;

  private enum OS {
    // Even on Windows, the default compiler from cpptasks (gcc) uses .so as a shared lib extension
    WINDOWS("win32", "so"), LINUX("linux", "so"), MAC("darwin", "dylib"), SOLARIS("solaris", "so");
    public final String name, libExtension;

    private OS(String name, String libExtension) {
      this.name = name;
      this.libExtension = libExtension;
    }
  }

  private static String arch() {
    return System.getProperty("os.arch");
  }

  private static OS os() {
    String osName = System.getProperty("os.name");
    if (osName.contains("Linux")) {
      return OS.LINUX;
    } else if (osName.contains("Mac")) {
      return OS.MAC;
    } else if (osName.contains("Windows")) {
      return OS.WINDOWS;
    } else if (osName.contains("Solaris") || osName.contains("SunOS")) {
      return OS.SOLARIS;
    } else {
      throw new UnsupportedOperationException("Unsupported operating system: "
          + osName);
    }
  }

  private static String resourceName() {
    OS os = os();
    String packagePrefix = Native.class.getPackage().getName().replace('.', '/');

    return "/" + packagePrefix + "/" + os.name + "/" + arch() + "/liblz4-java." + os.libExtension;
  }

  private static boolean loaded = false;
  private static boolean failedToLoad = false;

  public static synchronized boolean isLoaded() {
    return loaded;
  }

  public static boolean failedToLoad() {
    return failedToLoad;
  }

  private static void cleanupOldTempLibs() {
    String tempFolder = new File(System.getProperty("java.io.tmpdir")).getAbsolutePath();
    File dir = new File(tempFolder);

    File[] tempLibFiles = dir.listFiles(new FilenameFilter() {
      private final String searchPattern = "liblz4-java-";
      public boolean accept(File dir, String name) {
        return name.startsWith(searchPattern) && !name.endsWith(".lck");
      }
    });
    if(tempLibFiles != null) {
      for(File tempLibFile : tempLibFiles) {
        File lckFile = new File(tempLibFile.getAbsolutePath() + ".lck");
        if(!lckFile.exists()) {
          try {
            tempLibFile.delete();
          }
          catch(SecurityException e) {
            System.err.println("Failed to delete old temp lib" + e.getMessage());
          }
        }
      }
    }
  }

  public static synchronized void load() {
    load(NativeParameters.extractionPath);
  }

  public static synchronized void load(Path extractionPath) {
    if (loaded)
      return;
    if (failedToLoad)
      throw new UnsatisfiedLinkError("Native LZ4 dynamic library loading failed already. Not retrying.");

    if (os() == OS.LINUX) {
      // Try to load lz4-java (liblz4-java.so on Linux) from the java.library.path.
      try {
        System.loadLibrary("lz4-java");
        loaded = true;
        return;
      } catch (UnsatisfiedLinkError ex) {
        // Doesn't exist, so proceed to loading bundled library.
      }
    }

    String resourceName = resourceName();
    URL libraryUrl = Native.class.getResource(resourceName);
    if (libraryUrl == null)
      throw new UnsupportedOperationException("Unsupported OS/arch, cannot find " + resourceName + ". Please try building from source.");

    try {
      Path tempLib = null, tempLibLock = null, out;
      boolean extract = true;

      try {
        if (extractionPath == null) {
          cleanupOldTempLibs();

          // Create the .lck file first to avoid a race condition
          // with other concurrently running Java processes using lz4-java.
          out = tempLib = Files.createTempFile("liblz4-java-", "." + os().libExtension);
          tempLibLock = tempLib.resolveSibling(tempLib.getFileName().toString() + ".lck");
        } else {
          out = extractionPath.resolve("liblz4-java." + os().libExtension);
          if (Files.exists(out)) {
            byte[] sourceHash = ChecksumUtil.computeSum(libraryUrl);
            byte[] targetHash = ChecksumUtil.computeSum(out);
            extract = !Arrays.equals(sourceHash, targetHash);
          }
        }
        if (extract)
          copy(libraryUrl, out);
        System.load(out.toString());
        loaded = true;
      } finally {
        if (tempLib != null && Files.exists(tempLib)) {
          if (!loaded) {
            Files.deleteIfExists(tempLib);
            if (tempLibLock != null) {
              Files.deleteIfExists(tempLibLock);
            }
          } else {
            final String keepEnv = System.getenv("LZ4JAVA_KEEP_TEMP_JNI_LIB");
            final String keepProp = System.getProperty("lz4java.jnilib.temp.keep");
            if ((keepEnv == null || !keepEnv.equals("true")) &&
                (keepProp == null || !keepProp.equals("true")))
                // This probably doesn't work on Windows at all.
                tempLib.toFile().deleteOnExit();
            tempLibLock.toFile().deleteOnExit();
          }
        }
      }
    } catch (UnsatisfiedLinkError e) {
      failedToLoad = true;
      throw e;
    } catch (IOException e) {
      failedToLoad = true;
      throw new ExceptionInInitializerError(e);
    }
  }

  private static long copy(URL url, Path to) throws IOException {
    try (InputStream in = url.openStream()) {
      return Files.copy(in, to, StandardCopyOption.REPLACE_EXISTING);
    }
  }

}
