/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.ui.workbench.editor;

import org.eclipse.jface.resource.ImageDescriptor;
import org.simantics.db.ReadGraph;
import org.simantics.db.exception.DatabaseException;

/**
 * The purpose of an {@link EditorAdapter} is two-fold:
 * <ol>
 * <li>Checking whether an action can be performed on an input object (see
 * {@link #canHandle(ReadGraph, Object)}</li>
 * <li>Performing the action on that input object (see
 * {@link #openEditor(Object)}</li>
 * </ol>
 * An action can be anything, but for the purposes this interface is intended
 * for, the action is most often <em>opening an editor</em>.
 * {@link #canHandle(ReadGraph, Object)} is performed within a graph database
 * read transaction to make data model based decision making easier.
 * 
 * <p>
 * #openEditor(Object) shall be called only if
 * {@link #canHandle(ReadGraph, Object)} returns true for the same input. It is
 * possible that the data model has changed between invocations of
 * {@link #canHandle(ReadGraph, Object)} and {@link #openEditor(Object)} so
 * {@link #openEditor(Object)} implementations should always check the input
 * throughly instead of assuming that the data model state in
 * {@link #canHandle(ReadGraph, Object)} still holds.
 * 
 * Clients should extend at least {@link AbstractEditorAdapter} instead of
 * implementing this interface. This allows the extension point handler to set
 * the priority of the adapter according to what is described in extensions. The
 * only things that really need to be implemented are
 * {@link #canHandle(ReadGraph, Object)} and {@link #openEditor(Object)}.
 * 
 * <p>
 * These classes are instantiated by {@link EditorRegistry} from Eclipse
 * extensions so make sure that your implementations have a default constructor
 * available, or no constructors at all.
 * </p>
 * 
 * @author Tuukka Lehtonen
 * 
 * @see AbstractEditorAdapter
 * @see AbstractResourceEditorAdapter
 * @see SimpleEditorAdapter
 * @see IEditorRegistry the front-end interface for handling these classes
 * @see EditorRegistry an Eclipse extension point implementation of
 *      IResourceEditorRegistry
 * @see Prioritized
 */
public interface EditorAdapter {

    /**
     * @return
     */
    String getName();

    /**
     * @param r
     * @return
     */
    ImageDescriptor getIcon();

    /**
     * @return
     */
    int getPriority();

    /**
     * @param input the input of the editor
     * @return <code>true</code> if this adapter can open an editor for the
     *         specified input object
     */
    boolean canHandle(ReadGraph g, Object input) throws DatabaseException;

    /**
     * @param input the input object for the editor to be opened
     * @throws Exception PartInitException, or anything else that may happen
     *         while performing the action may be thrown
     */
    void openEditor(Object input) throws Exception;
    
    /**
     * Returns id of the editor.
     * @return id of the editor or null.
     */
    default String getEditorId() { return null; }

}
