/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.ui.workbench;

import java.lang.ref.Reference;
import java.lang.ref.SoftReference;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.simantics.Simantics;
import org.simantics.db.RequestProcessor;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.common.ResourceArray;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.RuntimeDatabaseException;
import org.simantics.db.service.LifecycleSupport;
import org.simantics.db.service.SerialisationSupport;

/**
 * @author Tuukka Lehtonen
 */
final class ResourceInputs {

    public static <T> Reference<T> makeReference(T t) {
        return new SoftReference<T>(t);
    }

    public static Resource resolveResource(RequestProcessor processor, String resourceId) throws DatabaseException {
        try {
            SerialisationSupport ss = processor.getService(SerialisationSupport.class);
            return ss.getResource(Long.parseLong(resourceId));
        } catch (NumberFormatException e) {
            throw new DatabaseException(e);
        }
    }

    public static ResourceArray makeResourceArray(RequestProcessor processor, Collection<String> resourceIds) throws DatabaseException {
        try {
            SerialisationSupport ss = processor.getService(SerialisationSupport.class);
            List<Resource> rs = new ArrayList<Resource>();
            for (String id : resourceIds)
                rs.add(ss.getResource(Long.parseLong(id)));
            return new ResourceArray(rs);
        } catch (NumberFormatException e) {
            throw new DatabaseException(e);
        }
    }

    public static String getRandomAccessId(Resource r) {
        try {
            SerialisationSupport support = getSession().getService(SerialisationSupport.class);
            return String.valueOf( support.getRandomAccessId(r) );
        } catch (DatabaseException e) {
            throw new RuntimeDatabaseException(e);
        }
    }

    public static List<String> getRandomAccessIds(ResourceArray ra) {
        try {
            List<String> randoms = new ArrayList<String>();
            SerialisationSupport support = getSession().getService(SerialisationSupport.class);
            for (Resource r : ra.resources)
                randoms.add(String.valueOf(support.getRandomAccessId(r)));
            return Collections.unmodifiableList(randoms);
        } catch (DatabaseException e) {
            throw new RuntimeDatabaseException(e);
        }
    }

    /**
     * @return a graph instance if it exists and has not yet been disposed,
     *         <code>null</code> otherwise
     */
    public static Session getSession() {
        Session s = Simantics.getSession();
        if (s.getService(LifecycleSupport.class).isClosed())
            throw new IllegalStateException("database session is closed");
        return s;
    }

    /**
     * @return a graph instance if it exists and has not yet been disposed,
     *         <code>null</code> otherwise
     */
    public static Session peekSession() {
        Session s = Simantics.peekSession();
        if (s == null)
            return null;
        if (s.getService(LifecycleSupport.class).isClosed())
            throw new IllegalStateException("database session is closed");
        return s;
    }

}
