/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.ui.workbench;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.ui.IElementFactory;
import org.eclipse.ui.IMemento;
import org.simantics.utils.ui.ErrorLogger;
import org.simantics.utils.ui.workbench.StringMemento;

/**
 * Factory for restoring a <code>ResourceEditorInput2</code>. The stored
 * representation of a <code>ResourceEditorInput2</code> remembers a reference
 * to the input resource, which is a string ID, a model URI and an <a
 * href="https://www.simantics.org/wiki/index.php/Terminology#R">RVI</a>.
 * 
 * <p>
 * The workbench will automatically create instances of this class as required.
 * It is not intended to be instantiated or subclassed by the client.
 * </p>
 */
public class ResourceEditorInputFactory2 implements IElementFactory {
    /**
     * Factory id. The workbench plug-in registers a factory by this name
     * with the "org.eclipse.ui.elementFactories" extension point.
     */
    private static final String ID_FACTORY = "org.simantics.ui.workbench.resourceEditorInputFactory2"; //$NON-NLS-1$

    /**
     */
    public static final String TAG_RESOURCE_ID = ResourceEditorInputFactory.TAG_RESOURCE_ID; //$NON-NLS-1$

    /**
     */
    public static final String TAG_EDITOR_ID = ResourceEditorInputFactory.TAG_EDITOR_ID; //$NON-NLS-1$

    /**
     * 
     */
    public static final String TAG_MODEL_ID = "modelId";

    /**
     * 
     */
    public static final String TAG_RVI = "rvi";

    /** Tag ID for node that contains external data */
    public static final String TAG_EXTERNAL_MEMENTO_ID = "external"; //$NON-NLS-1$

    /**
     * Creates a new factory.
     */
    public ResourceEditorInputFactory2() {
    }

    /* (non-Javadoc)
     * Method declared on IElementFactory.
     */
    @Override
    public IAdaptable createElement(IMemento memento) {
        // Get the resource id array.
        IMemento[] children = memento.getChildren(TAG_RESOURCE_ID);
        if (children.length != 1)
            return null;

        List<String> ids = new ArrayList<String>();
        for (IMemento child : children)
            ids.add(child.getTextData());

        String editorId = memento.getString(TAG_EDITOR_ID);
        String modelId = memento.getString(TAG_MODEL_ID);
        String rvi = memento.getString(TAG_RVI);

        try {
            ResourceEditorInput2 result = new ResourceEditorInput2(editorId, ids.iterator().next(), modelId, rvi);

            // Read external persistent memento
            String externalMementoStr = memento.getString(TAG_EXTERNAL_MEMENTO_ID);
            if (externalMementoStr != null) {
                StringMemento sm = new StringMemento(externalMementoStr);
                sm.writeToMemento(result.getPersistentStore());
            }
            return result;
        } catch (NumberFormatException e) {
            return null;
        } catch (IllegalArgumentException e) {
            ErrorLogger.defaultLogError(e);
            return null;
        }
    }

    /**
     * Returns the element factory id for this class.
     * 
     * @return the element factory id
     */
    public static String getFactoryId() {
        return ID_FACTORY;
    }

}
