/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.ui.workbench;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.ui.IElementFactory;
import org.eclipse.ui.IMemento;
import org.simantics.utils.ui.ErrorLogger;
import org.simantics.utils.ui.workbench.StringMemento;


/**
 * Factory for restoring a <code>ResourceEditorInput</code>.
 * The stored representation of a <code>ResourceEditorInput</code> remembers
 * a reference to the input resource, which is a string ID.
 * 
 * <p>
 * The workbench will automatically create instances of this class as required.
 * It is not intended to be instantiated or subclassed by the client.
 * </p>
 */
public class ResourceEditorInputFactory implements IElementFactory {
    /**
     * Factory id. The workbench plug-in registers a factory by this name
     * with the "org.eclipse.ui.elementFactories" extension point.
     */
    private static final String ID_FACTORY = "org.simantics.ui.workbench.resourceEditorInputFactory"; //$NON-NLS-1$

    /**
     */
    public static final String TAG_RESOURCE_ID = "resourceId"; //$NON-NLS-1$

    /**
     */
    public static final String TAG_EDITOR_ID = "editorId"; //$NON-NLS-1$

    /** Tag ID for node that contains external data */
    public static final String TAG_EXTERNAL_MEMENTO_ID = "external"; //$NON-NLS-1$

    /**
     * Creates a new factory.
     */
    public ResourceEditorInputFactory() {
    }

    /* (non-Javadoc)
     * Method declared on IElementFactory.
     */
    public IAdaptable createElement(IMemento memento) {
        // Get the resource id array.
        IMemento[] children = memento.getChildren(TAG_RESOURCE_ID);
        if (children.length == 0)
            return null;

        List<String> ids = new ArrayList<String>();
        for (IMemento child : children)
            ids.add(child.getTextData());

        String editorId = memento.getString(TAG_EDITOR_ID);

        try {
            ResourceEditorInput result = new ResourceEditorInput(editorId, ids);

            // Read external persistent memento
            String externalMementoStr = memento.getString(TAG_EXTERNAL_MEMENTO_ID);
            if (externalMementoStr != null) {
                StringMemento sm = new StringMemento(externalMementoStr);
                sm.writeToMemento(result.getPersistentStore());
            }
            return result;
        } catch (NumberFormatException e) {
            return null;
        } catch (IllegalArgumentException e) {
            ErrorLogger.defaultLogError(e);
            return null;
        }
    }

    /**
     * Returns the element factory id for this class.
     * 
     * @return the element factory id
     */
    public static String getFactoryId() {
        return ID_FACTORY;
    }

}
