package org.simantics.ui.workbench;

import java.lang.ref.Reference;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.PlatformObject;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IPersistableElement;
import org.simantics.Simantics;
import org.simantics.db.AsyncRequestProcessor;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.common.ResourceArray;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.exception.AdaptionException;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ResourceNotFoundException;
import org.simantics.db.layer0.exception.MissingVariableException;
import org.simantics.db.layer0.variable.RVI;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.db.request.Read;
import org.simantics.db.service.LifecycleSupport;
import org.simantics.ui.icons.ImageDescriptorProvider;
import org.simantics.ui.workbench.IResourceEditorInput;
import org.simantics.ui.workbench.ResourceInputs;
import org.simantics.ui.workbench.TitleRequest;
import org.simantics.ui.workbench.ToolTipRequest;
import org.simantics.utils.datastructures.cache.ProvisionException;
import org.simantics.utils.ui.ErrorLogger;
import org.simantics.utils.ui.workbench.StringMemento;

public class MultiResourceEditorInput extends PlatformObject implements IEditorInput, IResourceEditorInput, IPersistableElement {
    private final static boolean          DEBUG_EXISTS    = false;
    private final static boolean          DEBUG_UPDATE    = false;

    private static final String           NO_NAME         = "(no name)";
    
    private String editorID;
    private Reference<Resource> model;
    private String modelId;
    private List<String> resourceIds;
    private List<Reference<Resource>> resources;
    private List<String> rvis;
    
    private transient String              name;
    private transient boolean             exists;
    private transient String              tooltip;
    private transient ImageDescriptor     imageDesc;
    
    /** Persistent memento for external data */
    private final StringMemento           persistentStore = new StringMemento();
    
    public MultiResourceEditorInput(String editorID, Resource model, List<Resource> resources, List<RVI> rvis) {
        super();
        this.editorID = editorID;
        if (rvis != null && resources.size() != rvis.size())
            throw new IllegalArgumentException("There must be same amount of resources and RVIs");
        this.modelId = ResourceInputs.getRandomAccessId(model);
        this.model = ResourceInputs.makeReference(model);
        this.resourceIds = new ArrayList<>(resources.size());
        this.resources = new ArrayList<>(resources.size());
        for (Resource resource : resources) {
            this.resourceIds.add(ResourceInputs.getRandomAccessId(resource));
            this.resources.add(ResourceInputs.makeReference(resource));
        }
        if (rvis != null) {
            this.rvis = new ArrayList<>(rvis.size());
            for (RVI rvi : rvis) {
                this.rvis.add(rvi.toString());
            }
        }
        setNonExistant();
    }
    
    public MultiResourceEditorInput(String editorID, String modelId, List<String> resourceIds, List<String> rvis) {
        super();
        this.editorID = editorID;
        if (rvis != null && resourceIds.size() != rvis.size())
            throw new IllegalArgumentException("There must be same amount of resources and RVIs");
        this.modelId = modelId;
        this.model = null;
        this.resourceIds = new ArrayList<>(resourceIds.size());
        this.resources = new ArrayList<>(resourceIds.size());
        for (String resourceId : resourceIds) {
        	this.resourceIds.add(resourceId);
            this.resources.add(null);
        }
        this.rvis = rvis;
        setNonExistant();
        
    }
    
    @Override
    public void dispose() {
        model = null;
        resources = null;
    }
    
    @Override
    public void init(IAdaptable adapter) throws DatabaseException {
        Resource r = getResource();
        if (r != null)
            updateCaches(getSession(), true);
        
    }
    

    /**
     * @return a graph instance if it exists and has not yet been disposed,
     *         <code>null</code> otherwise
     */
    public Session getSession() {
        Session s = Simantics.getSession();
        if (s.getService(LifecycleSupport.class).isClosed())
            throw new IllegalStateException("database session is closed");
        return s;
    }
    
    @Override
    public boolean exists() {
        return exists;
    }
    
    @Override
    public boolean exists(ReadGraph graph) throws DatabaseException {
        try {
            assertExists(graph);
            return true;
        } catch (MissingVariableException e) {
        } catch (ResourceNotFoundException e) {
        } catch (Nonexistant e) {
        }
        return false;
    }
    
    public Resource getResource(int index) throws DatabaseException {
        Resource r = tryGetResource(index);
        if (r != null)
            return r;

        Session s = ResourceInputs.peekSession();
        if (s == null)
            return null;

        r = ResourceInputs.resolveResource( s, resourceIds.get(index) );
        while (resources.size() <= index)
            resources.add(null);
        this.resources.set(index, ResourceInputs.makeReference( r ));
        return r;
    }

    public Resource getModel0() throws DatabaseException {
        Resource r = tryGetModel();
        if (r != null)
            return r;

        Session s = ResourceInputs.peekSession();
        if (s == null)
            return null;

        r = ResourceInputs.resolveResource( s, modelId );
        this.model = ResourceInputs.makeReference( r );
        return r;
    }
    
    @Override
    public Resource getResource() {
        try {
            return getResource(0);
        } catch (DatabaseException e) {
            ErrorLogger.defaultLogError(e);
            return null;
        }
    }
    
    @Override
    public ResourceArray getResourceArray() {
        List<Resource> _resources = new ArrayList<>(resourceIds.size());
        try {
            for (int i = 0; i < resourceIds.size(); i++) {
                Resource r = getResource(i);
                if (r == null)
                    continue;
                _resources.add(r);
            }
        } catch (DatabaseException e) {
            ErrorLogger.defaultLogError(e);
            return ResourceArray.EMPTY;
        }
        ResourceArray ra = new ResourceArray(_resources);
        return ra;
    }
    
    public Resource getModel(ReadGraph graph) {
        try {
            return getModel0();
        } catch (DatabaseException e) {
            ErrorLogger.defaultLogError(e);
            return null;
        }
    }
    
    public String getRVI(int index) {
        return rvis.get(index);
    }
    
    public List<String> getRvis() {
		return rvis;
	}
    
    public int size() {
        return resourceIds.size();
    }
    
    
    /* (non-Javadoc)
     * @see org.eclipse.ui.IEditorInput#getImageDescriptor()
     */
    @Override
    public ImageDescriptor getImageDescriptor() {
        return imageDesc;
    }

    /* (non-Javadoc)
     * @see org.eclipse.ui.IEditorInput#getName()
     */
    @Override
    public String getName() {
        return name;
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.ui.IEditorInput#getToolTipText()
     */
    @Override
    public String getToolTipText() {
        return tooltip;
    }

    /* (non-Javadoc)
     * @see org.eclipse.ui.IEditorInput#getPersistable()
     */
    @Override
    public IPersistableElement getPersistable() {
        // Don't allow persistability when it's not possible.
        if (!isPersistable())
            return null;
        return this;
    }
    
    protected boolean isPersistable() {
        Session session = Simantics.peekSession();
        if (session == null)
            return false;
        LifecycleSupport lc = session.peekService(LifecycleSupport.class);
        if (lc == null)
            return false;
        if (lc.isClosed())
            return false;
        return true;
    }

    
    
    @Override
    public String getFactoryId() {
        return MultiResourceEditorInputFactory.getFactoryId();
    }
    
    @Override
    public void saveState(IMemento memento) {
        for (String resourceId : resourceIds) {
            IMemento child = memento.createChild(MultiResourceEditorInputFactory.TAG_RESOURCE_ID);
            child.putTextData(resourceId);
        }
        for (String rvi: rvis) {
            IMemento child = memento.createChild(MultiResourceEditorInputFactory.TAG_RVI);
            child.putTextData(rvi);    
        }
        memento.putString(MultiResourceEditorInputFactory.TAG_EDITOR_ID, editorID);
        memento.putString(MultiResourceEditorInputFactory.TAG_MODEL_ID, modelId);
        
        memento.putString(MultiResourceEditorInputFactory.TAG_EXTERNAL_MEMENTO_ID, persistentStore.toString());
    }
    
    @Override
    public <T> T getAdapter(Class<T> adapter) {
        //System.out.println("[ResourceEditorInput] getAdapter: " + adapter.getName());
        return null;
    }
    
    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + editorID.hashCode();
        // modelURI and rvi may change => can't use either in hashcode.
//        result = prime * result + ObjectUtils.hashCode(modelURI);
//        result = prime * result + ObjectUtils.hashCode(rvi);
        result = prime * result + Objects.hashCode(modelId);
        for (String resourceId : resourceIds)
            result = prime * result + Objects.hashCode(resourceId);
        return result;
    }
    
    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        MultiResourceEditorInput other = (MultiResourceEditorInput) obj;
        if (!editorID.equals(other.editorID))
            return false;
        if (!Objects.equals(modelId, other.modelId))
            return false;
        if (rvis.size() != other.rvis.size())
            return false;
        // TODO : should we allow random order.
        for (int i = 0; i < rvis.size(); i++) {
            if (!Objects.equals(rvis.get(i), other.rvis.get(i)))
                return false;
        }
        if (resourceIds.size() != other.resourceIds.size())
            return false;
        // TODO : should we allow random order.
        for (int i = 0; i < resourceIds.size(); i++) {
            if (!Objects.equals(resourceIds.get(i), other.resourceIds.get(i)))
                return false;
        }
        return true;
    }
    
    private void updateCaches(AsyncRequestProcessor processor, boolean sync) throws DatabaseException {
        ReadRequest req = new ReadRequest() {
            @Override
            public void run(ReadGraph g) throws DatabaseException {
                update(g);
            }
        };
        if (sync) {
            processor.syncRequest(req);
        } else {
            processor.asyncRequest(req);
        }
    }
    
    /* (non-Javadoc)
     * @see org.simantics.ui.workbench.IResourceEditorInput#update(org.simantics.db.Graph)
     */
    @Override
    public void update(ReadGraph g) throws DatabaseException {
        Resource r = getResource();
        if (r == null)
            return;

        if (DEBUG_UPDATE)
            System.out.println("update(" + this + ")");

        try {
            //assertExists(g);

            name = g.syncRequest(new TitleRequest(editorID, this));
            if (name == null)
                name = NO_NAME;

            tooltip = g.syncRequest(new ToolTipRequest(editorID, this));
            if (tooltip == null)
                tooltip = NO_NAME;

            try {
                ImageDescriptorProvider idp = g.adapt(r, ImageDescriptorProvider.class);
                imageDesc = idp.get();
            } catch (AdaptionException e) {
                imageDesc = ImageDescriptor.getMissingImageDescriptor();
            } catch (ProvisionException e) {
                imageDesc = ImageDescriptor.getMissingImageDescriptor();
                ErrorLogger.defaultLogError(e);
            }

            if (DEBUG_UPDATE)
                System.out.println("update(" + this + ") finished");
        } catch (DatabaseException e) {
            if (DEBUG_UPDATE)
                e.printStackTrace();
            setNonExistant();
        }
    }
    
    private void assertExists(ReadGraph g) throws DatabaseException {
        if (DEBUG_EXISTS)
            System.out.println("ResourceEditorInput2.assertExists(" + this + ") begins");

        // 1. Check resource existence
        Resource r = getResource();
        if (r == null)
            throw new Nonexistant();

        exists = g.hasStatement(r);
        if (!exists)
            throw new Nonexistant();

        // 2. Check model existence
        Resource model = getModel(g);
        if (model == null)
            throw new Nonexistant();

        exists = g.hasStatement(model);
        if (!exists)
            throw new Nonexistant();

        // 3. Validate rvi
        if (DEBUG_EXISTS)
            System.out.println("validating rvis: '" + rvis + "'");

        for (String rvi : rvis) {
            if(rvi != null && !rvi.isEmpty()) {
                Variable context = Variables.getPossibleConfigurationContext(g, model);
                if (context == null)
                    throw new Nonexistant();
                RVI rvi_ = RVI.fromResourceFormat(g, rvi);
                Variable variable = rvi_.resolvePossible(g, context);
                if (variable == null)
                    throw new Nonexistant();
            }
        }

        // Touch the diagram title calculation within this existence
        // checking request.
        g.syncRequest(new TitleRequest(editorID, this));

        if (DEBUG_EXISTS)
            System.out.println("ResourceEditorInput2.assertExists(" + this + ") finished");
    }

    private void setNonExistant() {
        if (DEBUG_UPDATE)
            System.out.println("setNonExistant(" + this + " @ " + System.identityHashCode(this) + ")");
        
        exists = false;
        tooltip = name = NO_NAME;
        imageDesc = ImageDescriptor.getMissingImageDescriptor();
    }
    
    public IMemento getPersistentStore() {
        return persistentStore;
    }
    
    /**
     * @see org.simantics.ui.workbench.IResourceEditorInput2#getVariable()
     */
    public Variable getVariable(int index) throws DatabaseException {
        return getSession().syncRequest(new Read<Variable>() {
            @Override
            public Variable perform(ReadGraph graph) throws DatabaseException {
                return getVariable(graph,index);
            }
        });
    }
    /**
     * @see org.simantics.ui.workbench.IResourceEditorInput2#getVariable(org.simantics.db.ReadGraph)
     */
    public Variable getVariable(ReadGraph graph, int index) throws DatabaseException {
        Resource model = getModel(graph);
        String rvi = getRVI(index);
        // Model + RVI
        if (rvi != null) {
            Variable configuration = Variables.getConfigurationContext(graph, model);
            RVI rrvi = RVI.fromResourceFormat(graph, rvi);
            return rrvi.resolve(graph, configuration);
        }
        // Absolute URI
        else {
            return Variables.getVariable(graph, model);
        }
    }
    
    private Resource tryGetResource(int index) {
        Reference<Resource> ref = resources.get(index);
        return ref == null ? null : ref.get();
    }

    private Resource tryGetModel() {
        Reference<Resource> ref = model;
        return ref == null ? null : ref.get();
    }
    
    static class Nonexistant extends DatabaseException {
        private static final long serialVersionUID = 9062837207032093570L;

        @Override
        public synchronized Throwable fillInStackTrace() {
            return this;
        }
    }
}
