package org.simantics.ui.toolbar;

import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.ToolBar;

/**
 * Utility to handle toolbar item scaling on non fractional display scales.
 * 
 * Combo and other text including widgets do not scale well with fractional scales. Examples: 100% and 200% scales work, 125% and 150% do not.
 * 
 * @author MarkoLuukkainen
 *
 */
public class ToolbarScaleUtil {
	
	// values for fix fractional scaling. see initScale method.
    private boolean initScale = false;
    private int scaledHeight = -1;
	private Font scaledFont;
	
	private Map<Control,Font> originalFonts = new HashMap<>();
	
	/**
	 * Initializes scale util.
	 * 
	 * @param parent
	 */
	public void initScale(Composite parent) {
		if (initScale)
			return;
		ToolBar toolBar = getToolBar(parent);
		// reading height from toolbar is unreliable, since parent may not have any content. 22 seems to be default value.
		int totalHeight = 22;//parent.getSize().y;
		// We need to separate total height and item height (without margins). item height is 16, which is default icon size.
		int actHeight = 16;
		
		// get current scale.
		int dpi = org.eclipse.swt.widgets.Display.getCurrent().getDPI().x;
		double scale = ((double)dpi)/96.0;
		// calculate fractional part.
		while (scale >= 1.99)
			scale -= 1.0;
		// if fractional part is zero, we do not need custom scaling.
		if (Math.abs(1.0-scale) < 0.1)
			return;
		scaledHeight = (int)(Math.ceil(actHeight / scale)+0.5);
		scaledHeight += (totalHeight-actHeight);
		
		if (toolBar != null) {
			// Create scaled font
			Font font = toolBar.getFont();
			if (font != null && scale > 1.0) {
				FontData data[] = font.getFontData();
				if (data.length == 1) {
					FontData newData = new FontData(data[0].getName(), data[0].getHeight(), data[0].getStyle());
					newData.height /= scale;
					scaledFont = new Font(Display.getCurrent(), newData);
				}
			}		
		}
		initScale = true;
	}
	
	public boolean isInitScale() {
		return initScale;
	}
	/**
	 * Returns scaled height or value below zero.
	 * 
	 */
	public int getScaledHeight() {
		return scaledHeight;
	}
	
	/**
	 * Returns scaled font or null.
	 * @return
	 */
	public Font getScaledFont() {
		return scaledFont;
	}

	private ToolBar getToolBar(Composite parent) {
		Composite c = parent;
    	while (c != null) {
    		if (c instanceof ToolBar)
    			return (ToolBar)c;
    		c = c.getParent();
    	}
    	return null;
    }
	
	public void scale(Control control) {
		scale(control, false);
	}
	
	public void scale(Control control, boolean restoreOnDispose) {
		if (scaledFont == null)
			return;
		Font currentFont = control.getFont();
		if (currentFont == scaledFont)
			return;
		if (restoreOnDispose) {
			if (currentFont != null) {
				originalFonts.put(control, currentFont);
			}
		}
		control.setFont(scaledFont);
	}
	
	public void scale(ToolBar toolbar) {
		// ToolItems of a ToolBar do not have individual font setting. Thus we need to set font to a ToolBar.
		// Note: it seems that default toolbar font is not picked correctly, so we have to override that.
		if (scaledFont == null)
			return;
		if (toolbar.getFont() == scaledFont)
			return;
		// ToolBars are usually reused. We want to keep and restore the original font, so that the ToolBar does not refer to a disposed custom font. 
		originalFonts.put(toolbar, toolbar.getFont());
		toolbar.setFont(scaledFont);
	}
	
	public void dispose() {
		for (Entry<Control, Font> e : originalFonts.entrySet()) {
			if (e.getKey().isDisposed())
				continue;
			e.getKey().setFont(e.getValue());
		}
		originalFonts.clear();
		if (scaledFont != null) {
			scaledFont.dispose();
			scaledFont = null;
		}
	}
}
