package org.simantics.ui.toolbar;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.commands.Command;
import org.eclipse.core.commands.State;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.commands.ICommandService;
import org.eclipse.ui.handlers.IHandlerService;
import org.eclipse.ui.handlers.RadioState;
import org.eclipse.ui.handlers.RegistryToggleState;
import org.simantics.utils.datastructures.MapList;

/**
 * Registry for storing command states separately for each IEditorPart
 * 
 * TODO : cleaning editor states and listeners is now implemented through ToolBarContributor. Is this OK?
 * 
 * @author Marko Luukkainen <marko.luukkainen@vtt.fi>
 *
 */
public class CommandStateRegistry {

	
	private static CommandStateRegistry instance;
	
	
	public static CommandStateRegistry getInstance() {
		if (instance == null)
			instance = new CommandStateRegistry();
		return instance;
	}
	
	
	
	private Map<IWorkbenchPart,Map<String,Boolean>> toggleStates = new HashMap<IWorkbenchPart, Map<String,Boolean>>();
	private Map<String,Boolean> defaultToggleStates = new HashMap<String, Boolean>();
	
	private Map<String,String> defaultRadioStates = new HashMap<String, String>();
	private Map<IWorkbenchPart,Map<String,String>> radioStates = new HashMap<IWorkbenchPart, Map<String,String>>();

	private ICommandService service;
	private IHandlerService handlerService;
	
	private CommandStateRegistry() {
		service = (ICommandService) PlatformUI.getWorkbench().getService(ICommandService.class);
		handlerService = (IHandlerService) PlatformUI.getWorkbench().getService(IHandlerService.class);
	}
	
	/**
	 * Stores default state for a command.
	 * 
	 * Note: uses current state as default. 
	 * 
	 * @param commandId
	 */
	public void storeDefaultState(String commandId) {
		Command command = service.getCommand(commandId);
		State toggleState = command.getState(RegistryToggleState.STATE_ID);
		State radioState = command.getState(RadioState.STATE_ID);
		if (toggleState != null) {
			if (!defaultToggleStates.containsKey(commandId))
				defaultToggleStates.put(commandId, getToggleState(command));
		} else if (radioState != null) {
			String value = (String) radioState.getValue();
			if (!defaultRadioStates.containsKey(commandId))
				defaultRadioStates.put(commandId, value);
		} else {
			throw new IllegalArgumentException("Command " + commandId + " does not have a state");
		}
	}
	/**
	 * Stores toggle state of a command.
	 * @param part
	 * @param commandId
	 * @param checked
	 */
	public void setEditorState(IWorkbenchPart part, String commandId, boolean checked) {
		Map<String,Boolean> editorStates = toggleStates.get(part);
		if (editorStates == null) {
			editorStates = new HashMap<String, Boolean>();
			toggleStates.put(part, editorStates);
		}
		editorStates.put(commandId, checked);
		fireStateChange(part, commandId, Boolean.toString(checked));
	}
	
	/**
	 * Stores radio state of a command.
	 * @param part
	 * @param commandId
	 * @param value
	 */
	public void setEditorState(IWorkbenchPart part, String commandId, String value) {
		Map<String,String> editorStates = radioStates.get(part);
		if (editorStates == null) {
			editorStates = new HashMap<String, String>();
			radioStates.put(part, editorStates);
		}
		editorStates.put(commandId, value);
		fireStateChange(part, commandId, value);
	}
	
	
	
	public Map<String, Boolean> getDefaultToggleStates() {
		return defaultToggleStates;
	}
	
	public Map<String, String> getDefaultRadioStates() {
		return defaultRadioStates;
	}
	
	public Map<String, Boolean> getEditorToggleStates(IWorkbenchPart part) {
		return toggleStates.get(part);
	}
	
	public Map<String, String> getEditorRadioStates(IWorkbenchPart part) {
		return radioStates.get(part);
	}
	
	public Boolean getToggleState(IWorkbenchPart part, String commandId) {
		if (part == null)
			return defaultToggleStates.get(commandId);
		Map<String,Boolean> editorStates = toggleStates.get(part);
		if (editorStates == null) {
			return defaultToggleStates.get(commandId);
		}
		return editorStates.get(commandId);
	}
	
	public String getRadioState(IWorkbenchPart part, String commandId) {
		if (part == null)
			return defaultRadioStates.get(commandId);
		Map<String,String> editorStates = radioStates.get(part);
		if (editorStates == null) {
			return defaultRadioStates.get(commandId);
		}
		return editorStates.get(commandId);
	}
	
	public void clearStates(IWorkbenchPart part) {
		toggleStates.remove(part);
		radioStates.remove(part);
		listenerWithPart.remove(part);
	}
	
	private boolean getToggleState(Command command) {
		State toggleState = command.getState(RegistryToggleState.STATE_ID);
		return (Boolean)toggleState.getValue();
	}
	
	
	private List<CommandStateListener> listeners = new ArrayList<CommandStateListener>();
	private MapList<String, CommandStateListener> listenerWithCommandId = new MapList<String, CommandStateListener>();
	private MapList<IWorkbenchPart, CommandStateListener> listenerWithPart = new MapList<IWorkbenchPart, CommandStateListener>();
	
	/**
	 * Attaches a listener that receives all state changes of all commands and all workbench parts.
	 * @param listener
	 */
	public void addListener(CommandStateListener listener) {
		if (!listeners.contains(listener))
			listeners.add(listener);
	}
	
	/**
	 * Attaches a listener to receive state changes of specific workbench part.
	 * @param part
	 * @param listener
	 */
	public void addListener(IWorkbenchPart part, CommandStateListener listener) {
		if (!listenerWithPart.contains(part,listener))
			listenerWithPart.add(part,listener);
	}
	
	/**
	 * Attaches a listener to receive state changes of specific command.
	 * @param part
	 * @param listener
	 */
	public void addListener(String commandId, CommandStateListener listener) {
		if (!listenerWithCommandId.contains(commandId,listener))
			listenerWithCommandId.add(commandId, listener);
	}
	
	/**
	 * Removes the listener from the registry.
	 * @param listener
	 */
	public void removeListener(CommandStateListener listener) {
		listeners.remove(listener);
		Set<String> commandIds = new HashSet<String>();
		commandIds.addAll(listenerWithCommandId.getKeys());
		for (String commandId : commandIds) {
			listenerWithCommandId.remove(commandId, listener);
		}
		commandIds.clear();
		Set<IWorkbenchPart> parts = new HashSet<IWorkbenchPart>();
		parts.addAll(listenerWithPart.getKeys());
		for (IWorkbenchPart part : parts) {
			listenerWithPart.remove(part, listener);
		}
		parts.clear();
	}
	
	private List<CommandStateListener> fireList = new ArrayList<CommandStateListener>();
	
	private void fireStateChange(IWorkbenchPart part, String commandId, String state) {
		fireList.clear();
		fireList.addAll(listeners);
		List<CommandStateListener> list = listenerWithCommandId.getValues(commandId);
		if (list != null) {
			fireList.addAll(list);
			
		}
		list = listenerWithPart.getValues(part);
		if (list != null) {
			fireList.addAll(list);
			
		}	
		for (CommandStateListener l : fireList) {
			l.stateChanged(part, commandId, state);
		}

	}
	
}
