/*******************************************************************************
 * Copyright (c) 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.ui.contribution;

import java.util.Arrays;
import java.util.Comparator;
import java.util.Hashtable;
import java.util.List;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExecutableExtension;
import org.eclipse.jface.action.ContributionItem;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.resource.LocalResourceManager;
import org.eclipse.jface.resource.ResourceManager;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.ui.ISelectionService;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.RequestProcessor;
import org.simantics.db.Resource;
import org.simantics.db.common.request.BinaryRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.procedure.Procedure;
import org.simantics.ui.selection.WorkbenchSelectionUtils;
import org.simantics.ui.workbench.action.ResourceEditorAdapterAction;
import org.simantics.ui.workbench.editor.EditorAdapter;
import org.simantics.ui.workbench.editor.EditorRegistry;
import org.simantics.utils.strings.AlphanumComparator;
import org.simantics.utils.ui.SWTUtils;
import org.simantics.utils.ui.workbench.WorkbenchUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Tuukka Lehtonen
 * @since 1.60.0
 */
public class OpenWithMenuContribution2 extends ContributionItem implements IExecutableExtension {

	private static final Logger LOGGER = LoggerFactory.getLogger(OpenWithMenuContribution2.class);

	private static record AMenu(MenuItem item, Menu menu) {}

	protected ResourceManager resourceManager;

	private String editorId;

	private boolean showEmptyMenu = true;

	/**
	 * Used to ensure that already old menu populations are not populated at all.
	 */
	private int modCount = 0;

	public OpenWithMenuContribution2() {
		resourceManager = new LocalResourceManager(JFaceResources.getResources());
	}

	@Override
	public void setInitializationData(IConfigurationElement config, String propertyName, Object data)
			throws CoreException {
		if ("class".equals(propertyName)) {
			if (data instanceof String params) {
				String[] parameters = params.split(";"); //$NON-NLS-1$
				for (String parameter : parameters) {
					String[] keyValue = parameter.split("="); //$NON-NLS-1$
					if (keyValue.length == 1) {
						// Assume this to be the editorId parameter
						setEditorId((String) keyValue[0]);
					} else if (keyValue.length == 2) {
						if ("showEmptyMenu".equals(keyValue[0])) {
							showEmptyMenu = Boolean.parseBoolean(keyValue[1]);
						}
					}
				}
			} else if (data instanceof Hashtable) {
				@SuppressWarnings("unchecked")
				Hashtable<String, String> params = (Hashtable<String, String>) data;
				setEditorId(params.get("editorId"));
				showEmptyMenu = Boolean.parseBoolean(params.get("showEmptyMenu"));
			}

			if ("activeEditorId".equals(this.editorId)) {
				setEditorId(WorkbenchUtils.getActiveWorkbenchPart().getSite().getId());
			}
		}
	}

	protected String getText() {
		return "Open With";
	}

	public String getEditorId() {
		return editorId;
	}

	public void setEditorId(String editorId) {
		this.editorId = editorId;
	}

	@Override
	public void dispose() {
		if (resourceManager != null) {
			resourceManager.dispose();
			resourceManager = null;
		}
		super.dispose();
	}

	@Override
	public boolean isDynamic() {
		return true;
	}

	protected IStructuredSelection getSelection() {
		IWorkbenchWindow window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
		ISelectionService service = window.getSelectionService();
		ISelection sel = service.getSelection();
		return sel instanceof IStructuredSelection ss ? ss : null;
	}

	protected static Object extractResource(RequestProcessor processor, Object object) throws DatabaseException {
		Resource resource = WorkbenchSelectionUtils.getPossibleResourceFromSelection(processor, object);
		return resource != null ? resource : object;
	}

	protected Object resolveInput() {
		IStructuredSelection selection = getSelection();

		// For backwards compatibility single-selections are pre-processed to resources.
		Object input = selection;
		if (selection != null && selection.size() == 1) {
			try {
				input = extractResource(Simantics.getSession(), selection.getFirstElement());
			} catch (DatabaseException e) {
				LOGGER.error("Failed to resolve input resource from {}", selection, e);
			}
		}

		return input;
	}

	@Override
	public void fill(Menu menu, int index) {
		Object input = resolveInput();
		if (input == null)
			return;

		int myCount = ++modCount;
		var display = menu.getDisplay();

		AMenu subMenu = showEmptyMenu
				? createSubMenu(menu, index, getText(), false)
				: new AMenu(null, menu);

		Simantics.getSession().asyncRequest(
				new ResolveEditorAdapters(input, editorId),
				new Procedure<>() {
					@Override
					public void execute(List<EditorAdapter> adapters) {
						if (adapters.isEmpty() && showEmptyMenu)
							return;
						SWTUtils.asyncExec(display,
								() -> contributeOpenWithMenuItems(subMenu, index, input, adapters, myCount));
					}
					@Override
					public void exception(Throwable t) {
						LOGGER.error("Resolution of suitable editor adapters for Open With menu failed.", t); //$NON-NLS-1$
					}
				});
	}

	private void contributeOpenWithMenuItems(AMenu parent, int index, Object input, List<EditorAdapter> adapters, int myCount) {
		if (adapters.isEmpty()
				|| myCount != modCount
				|| parent.menu().isDisposed()
				|| resourceManager == null)
			return;

		AMenu openWith = showEmptyMenu
				? parent
				: createSubMenu(parent.menu(), index, getText(), true);

		for (var adapter : adapters)
			addMenuItem(openWith.menu(), new Adapter(adapter, input, true));

		openWith.item.setEnabled(true);
	}

	private AMenu createSubMenu(Menu parent, int index, String label, boolean enabled) {
		MenuItem item = new MenuItem(parent, SWT.CASCADE, index);
		item.setText(label);
		item.setEnabled(enabled);
		Menu submenu = new Menu(parent);
		item.setMenu(submenu);
		return new AMenu(item, submenu);
	}

	protected static class ResolveEditorAdapters extends BinaryRead<Object, String, List<EditorAdapter>> {
		public ResolveEditorAdapters(Object input, String filteredEditorId) {
			super(input, filteredEditorId);
		}

		@Override
		public List<EditorAdapter> perform(ReadGraph graph) throws DatabaseException {
			LOGGER.trace(ResolveEditorAdapters.class.getName());
			return resolveEditorAdapters(graph, parameter, parameter2);
		}
	}

	private static final int compare(int p1, int p2, String t1, String t2) {
		int delta = Integer.compare(p1, p2);
		if (delta != 0)
			return delta;
		return AlphanumComparator.CASE_INSENSITIVE_COMPARATOR.compare(t1, t2);
	};

	private static final Comparator<EditorAdapter> EDITOR_ADAPTER_COMPARATOR =
			(o1, o2) -> compare(
					// Descending priority order
					o2.getPriority(),
					o1.getPriority(),
					ResourceEditorAdapterAction.makeName(o1),
					ResourceEditorAdapterAction.makeName(o2));

	private static List<EditorAdapter> resolveEditorAdapters(ReadGraph graph, Object input, String filteredEditorId) throws DatabaseException {
		var s = Arrays.asList(EditorRegistry.getInstance().getAdaptersFor(graph, input)).stream();
		if (filteredEditorId != null) {
			s = s.filter(a -> !filteredEditorId.equals(a.getEditorId()));
		}
		return s.sorted(EDITOR_ADAPTER_COMPARATOR)
				.toList();
	}

	private void addMenuItem(Menu parentMenu, Adapter adapter) {
		MenuItem item = new MenuItem(parentMenu, SWT.PUSH);
		String text = adapter.getText();
		if (LOGGER.isDebugEnabled()) {
			text = text + " (" + adapter.getAdapter().getClass().getCanonicalName() + ") [" + adapter.getPriority() + "]";
		}
		item.setText(text);
		ImageDescriptor descriptor = adapter.getImageDescriptor();
		if (descriptor != null) {
			item.setImage(resourceManager.createImage(descriptor));
		}
		item.addSelectionListener(adapter);
	}

	protected static class Adapter extends ResourceEditorAdapterAction implements SelectionListener {
		boolean remember;

		public Adapter(EditorAdapter adapter, Object r, boolean remember) {
			super(adapter, r);
			this.remember = remember;
		}

		@Override
		public void widgetDefaultSelected(SelectionEvent e) {
			widgetSelected(e);
		}

		@Override
		public void widgetSelected(SelectionEvent e) {
			run();
		}

		@Override
		protected void safeRun() throws Exception {
			super.safeRun();

			if (remember) {
				// Make this choice the default for the next time.
				EditorRegistry.getInstance().getMappings().put(getResource(), getAdapter());
			}
		}
	}

}
