/*******************************************************************************
 * Copyright (c) 2007, 2025 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - improvements
 *******************************************************************************/
package org.simantics.structural2.variables;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Statement;
import org.simantics.db.common.procedure.adapter.TransientCacheAsyncListener;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.NoSingleResultException;
import org.simantics.db.layer0.function.All;
import org.simantics.db.layer0.request.ClassificationsRequest;
import org.simantics.db.layer0.request.PossibleResource;
import org.simantics.db.layer0.variable.AbstractChildVariable;
import org.simantics.db.layer0.variable.NodeSupport;
import org.simantics.db.layer0.variable.StandardAssertedGraphPropertyVariable;
import org.simantics.db.layer0.variable.StandardConstantGraphPropertyVariable;
import org.simantics.db.layer0.variable.StandardGraphPropertyVariable;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.VariableNode;
import org.simantics.layer0.Layer0;
import org.simantics.simulator.variable.NodeManager;
import org.simantics.structural2.Functions;
import org.simantics.structural2.procedural.Connection;
import org.simantics.structural2.procedural.ConnectionPoint;
import org.simantics.structural2.procedural.Expression;
import org.simantics.structural2.procedural.Interface;
import org.simantics.structural2.procedural.Property;
import org.simantics.structural2.procedural.Terminal;
import org.simantics.structural2.procedural.UndefinedConnection;
import org.simantics.structural2.procedural.UndefinedProperty;
import org.simantics.utils.datastructures.Pair;
import org.slf4j.LoggerFactory;

import gnu.trove.map.hash.THashMap;

public class StandardProceduralChildVariable extends AbstractChildVariable {

	private static final org.slf4j.Logger LOGGER = LoggerFactory.getLogger(StandardProceduralChildVariable.class);

	/*
	 * Extension points
	 * 
	 */
	public Variable getPossibleSpecialChild(ReadGraph graph, String name) throws DatabaseException {
		return null;
	}

	public void collectSpecialChildren(ReadGraph graph, Map<String, Variable> children) throws DatabaseException {
	}

	/*
	 * Standard implementation
	 * 
	 */

	final protected String name;
	final protected Variable parent;
	final private Resource type;
	final private Map<String, Variable> properties;
	final private List<Object> propertyIdentity;

	public StandardProceduralChildVariable(ReadGraph graph, Variable parent, VariableNode node, String name, Resource type, List<Property> properties, Collection<Connection> conns) throws DatabaseException {
		super(node);
		assert name != null;
		assert type != null;
		this.name = name;
		this.parent = parent;
		this.type = type;
		this.properties = new THashMap<String,Variable>();

		this.propertyIdentity = new ArrayList<Object>(properties.size()+conns.size());
		propertyIdentity.addAll(properties);
		propertyIdentity.addAll(conns);

		Layer0 L0 = Layer0.getInstance(graph);

		Map<String,Statement> assertedProperties = new THashMap<String, Statement>();
		for(Statement stm : graph.getAssertedStatements(type, L0.HasProperty)) {
			String pn = graph.getRelatedValue(stm.getPredicate(), L0.HasName, Bindings.STRING);
			assertedProperties.put(pn, stm);
		}

		Collection<Object> nodeProperties = All.getPossibleNodeProperties(graph, this);
		Set<String> used = Collections.emptySet();
		if (nodeProperties != null && node != null && nodeProperties.size() > 0) {
			used = new HashSet<>(nodeProperties.size());
			@SuppressWarnings("unchecked")
			NodeSupport<Object> support = node.support;
			NodeManager<Object> manager = support.manager;
			for(Object nodeProperty : nodeProperties) {
				String pName = manager.getName(nodeProperty);
				used.add(pName);
				Statement assertedProperty = assertedProperties.get(pName); 
				if(assertedProperty != null) {
					this.properties.put(pName, new StandardAssertedGraphPropertyVariable(graph, this, new VariableNode<>(support, nodeProperty), assertedProperty.getSubject(), assertedProperty.getPredicate(), assertedProperty.getObject()));
				} else {
					String propertyUri = manager.getPropertyURI(node.node, nodeProperty);
					Resource predicate = propertyUri != null
							? graph.syncRequest(new PossibleResource(propertyUri), TransientCacheAsyncListener.instance())
									: null;
					this.properties.put(pName, new StandardGraphPropertyVariable(graph, this, new VariableNode<>(support, nodeProperty), predicate));
				}
			}
		}

		for(Map.Entry<String, Statement> entry : assertedProperties.entrySet()) {
			String pName = entry.getKey();
			if(used.contains(pName)) continue;
			Statement assertedProperty = entry.getValue();
			this.properties.put(pName, new StandardAssertedGraphPropertyVariable(graph, this, null, assertedProperty.getSubject(), assertedProperty.getPredicate(), assertedProperty.getObject()));
		}


		for(Property p : properties) {
			String pn = graph.getRelatedValue(p.relation, L0.HasName, Bindings.STRING);
			if (p instanceof UndefinedProperty) {
				this.properties.remove(pn);
				continue;
			}

			if(p.value == null) {
				LOGGER.error("StandardProceduralChildVariable " + getURI(graph) + ": null value for property " + pn);
			} else if (p.value instanceof Expression) {
				Expression expression = (Expression)p.value;
				this.properties.put(pn, new StructuralProceduralExpressionPropertyVariable(graph, this, p.relation, expression.text) );
			} else if (this.properties.containsKey(pn)) {
				// The property overrides the value of an asserted property variable
				StandardGraphPropertyVariable prop = (StandardGraphPropertyVariable)this.properties.get(pn);
				if (Objects.equals(prop.property.predicate, p.relation)) {
					this.properties.put(pn, new StandardProdeduralPropertyVariable(graph, this, prop, p.value));
				} else {
					LOGGER.warn("Ignored attempt to override asserted property {}/{}#{} with a different relation", parent.getURI(graph), name, pn);
				}
			} else {
				this.properties.put(pn, new StandardConstantGraphPropertyVariable(graph, this, p.relation, p.value) );
			}
		}
		Map<Resource,FixedConnection> map = new HashMap<Resource,FixedConnection>();
		for(Connection conn : conns) {
			Resource p = null;
			List<Pair<String,Resource>> cps = new ArrayList<Pair<String,Resource>>();
			for(ConnectionPoint cp : conn.connectionPoints) {
				if(cp instanceof Terminal) {
					Terminal t = (Terminal)cp;
					if(t.component.equals(name)) {
						p = t.relation;
						continue;
					}
					cps.add(Pair.make(t.component, t.relation));
				}
				if(cp instanceof Interface) {
					Interface inf = (Interface)cp;
					cps.add(new Pair<String,Resource>(null, inf.relation));
				}
			}
			if(p != null) {
				FixedConnection fc = map.get(p);
				if(fc == null) {
					fc = conn instanceof UndefinedConnection ? new org.simantics.structural2.variables.UndefinedConnection(parent) : new FixedConnection(parent);
					map.put(p, fc);
				}
				fc.addAll(cps);
			}
		}
		for(Map.Entry<Resource, FixedConnection> entry : map.entrySet()) {
			Resource cp = entry.getKey();
			String cpName = graph.getRelatedValue(cp, L0.HasName, Bindings.STRING);
			this.properties.put(cpName, new StandardConstantGraphPropertyVariable(graph, this, cp, entry.getValue()));
		}

	}

	@Override
	public void validate(ReadGraph graph) throws DatabaseException {
	}

	@Override
	public Resource getType(ReadGraph graph) throws DatabaseException {
		return type;
	}

	@Override
	public Resource getPossibleType(ReadGraph graph) throws DatabaseException {
		return type;
	}

	@Override
	public Resource getType(ReadGraph graph, Resource baseType) throws DatabaseException {
		if (graph.isInheritedFrom(type, baseType))
			return type;
		throw new NoSingleResultException("variable " + getPossibleURI(graph) + " has no type", -1);
	}

	@Override
	public Resource getPossibleType(ReadGraph graph, Resource baseType) throws DatabaseException {
		return graph.isInheritedFrom(type, baseType) ? type : null;
	}

	@Override
	protected Variable getPossibleDomainProperty(ReadGraph graph, String name) throws DatabaseException {
		return properties.get(name);
	}

	@Override
	public Variable getPossibleChild(ReadGraph graph, String name) throws DatabaseException {
		return Functions.structuralChildDomainChildren.getVariable(graph, this, name);
	}

	@Override
	public Map<String, Variable> collectDomainProperties(ReadGraph graph, Map<String, Variable> properties) throws DatabaseException {
		if(!this.properties.isEmpty()) {
			if(properties == null) properties = new THashMap<String,Variable>(this.properties.size());
			properties.putAll(this.properties);
		}
		return properties;
	}

	@Override
	public Collection<Variable> getChildren(ReadGraph graph) throws DatabaseException {
		Map<String,Variable> result = Functions.structuralChildDomainChildren.getVariables(graph, this, null);
		if(result == null) return Collections.emptyList();
		else return result.values();
	}

	public Set<String> getClassifications(ReadGraph graph) throws DatabaseException {
		Resource type = getPossibleType(graph);
		return (type != null)
				? graph.syncRequest(new ClassificationsRequest(Collections.singleton(type)))
						: Collections.<String>emptySet();
	}

	@Override
	public String getName(ReadGraph graph) throws DatabaseException {
		return name;
	}

	@Override
	public Variable getParent(ReadGraph graph) throws DatabaseException {
		return parent;
	}

	@Override
	final public Resource getRepresents(ReadGraph graph) throws DatabaseException {
		return null;
	}

	@Override
	public int hashCode() {
		return parent.hashCode() + 31*name.hashCode() + 41*type.hashCode() + 71*System.identityHashCode(propertyIdentity);
	}

	@Override
	public boolean equals(Object obj) {

		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;

		StandardProceduralChildVariable other = (StandardProceduralChildVariable) obj;

		if(!name.equals(other.name)) return false;
		if(!type.equals(other.type)) return false;
		if(!parent.equals(other.parent)) return false;
		if(propertyIdentity != other.propertyIdentity) return false;

		return true;

	}

}
