/*******************************************************************************
 * Copyright (c) 2025 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics;

import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.List;

import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Platform;
import org.eclipse.osgi.util.NLS;
import org.simantics.utils.strings.EString;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Tuukka Lehtonen
 * @since 1.63.0 
 */
public class Workspaces {

	private static final Logger LOGGER = LoggerFactory.getLogger(Workspaces.class);

	public static final String DEFAULT_VERSION_FILE = ".version"; //$NON-NLS-1$

	public static Path getWorkspacePath() {
		IPath finalPath = Platform.getLocation();
		if (finalPath == null)
			throw new IllegalStateException(Messages.Workspaces_CannotResolveWorkspacePathError);
		return finalPath.toFile().toPath();
	}

	private static String incompatibleWorkpaceMessage(Path ws, String expectedVersion, String gotVersion) {
		return NLS.bind(Messages.Workspaces_IncompatibleWorkspaceVersionError,
				new Object[] {
						ws.toAbsolutePath().toString(),
						expectedVersion,
						gotVersion,
				});
	}

	public static void assertWorkspaceVersion(String... expectedVersionDescriptor) throws PlatformException {
		Path ws = getWorkspacePath();
		Path ver = ws.resolve(DEFAULT_VERSION_FILE); //$NON-NLS-1$
		assertWorkspaceVersion(ws, ver, expectedVersionDescriptor);
	}

	public static void assertWorkspaceVersion(Path workspace, Path versionFile, String... expectedVersionDescriptor) throws PlatformException {
		try {
			String error = validateWorkspaceVersion(workspace, versionFile, expectedVersionDescriptor);
			if (error != null) {
				throw new UnsupportedWorkspaceVersionException(error);
			}
		} catch (IOException e) {
			LOGGER.error("Failed o validate workspace {} version file {} with expected version {}", workspace, versionFile, Arrays.toString(expectedVersionDescriptor), e);
			throw new UnsupportedWorkspaceVersionException("Unexpected I/O exception while checking workspace version information. Cannot safely commence program startup.", e); //$NON-NLS-1$
		}
	}

	public static String validateWorkspaceVersion(Path workspace, Path versionFile, String... expectedVersionDescriptor) throws IOException {
		Charset charset = StandardCharsets.UTF_8;
		try {
			List<String> lines = Files.readAllLines(versionFile, charset);
			boolean ok = true;
			if (lines.size() >= expectedVersionDescriptor.length) {
				for (int i = 0; i < expectedVersionDescriptor.length; ++i) {
					String got = lines.get(i);
					String expected = expectedVersionDescriptor[i];
					if (!got.equals(expected)) {
						ok = false;
						break;
					}
				}
			} else {
				ok = false;
			}
			return ok ?  null : incompatibleWorkpaceMessage(workspace, EString.implode(expectedVersionDescriptor, "\n"), EString.implode(lines, "\n")); //$NON-NLS-1$ //$NON-NLS-2$
		} catch (NoSuchFileException e) {
			Files.write(versionFile, Arrays.asList(expectedVersionDescriptor), charset);
			return null;
		}
	}

}