/*******************************************************************************
 * Copyright (c) 2013 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics;

import java.util.concurrent.Semaphore;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;

/**
 * A dummy database-family job that can be used for signaling that a large
 * database job is in progress although it is technically not running in a
 * {@link Job}.
 * 
 * <p>
 * To start the job and wait for it to start, use
 * {@link #scheduleAndWaitForRunning()}. To end the job and wait for it to die,
 * use {@link #disposeAndJoin()}.
 * 
 * @author Tuukka Lehtonen
 */
public class SleepingDatabaseJob extends DatabaseJob {

    private final Semaphore start = new Semaphore(0);
    private final Semaphore end = new Semaphore(0);

    public SleepingDatabaseJob(String name) {
        super(name);
    }

    @Override
    protected final IStatus run(IProgressMonitor monitor) {
        start.release();
        try {
            return work(monitor);
        } finally {
            try {
                end.acquire();
            } catch (InterruptedException e) {
                // Some other party wanted to kill the job. So be it.
            }
        }
    }

    protected IStatus work(IProgressMonitor monitor) {
        return Status.OK_STATUS;
    }

    public SleepingDatabaseJob scheduleAndWaitForRunning() throws InterruptedException {
        schedule();
        start.acquire();
        start.release();
        return this;
    }

    public void scheduleAndWaitForRunningUninterruptibly() {
        schedule();
        start.acquireUninterruptibly();
        start.release();
    }

    public void disposeAndJoin() throws InterruptedException {
        end.release();
        join();
    }

    /**
     * @param name
     * @param runnable
     * @throws InterruptedException
     */
    public static void sleepWhile(String name, Runnable runnable) throws InterruptedException {
        SleepingDatabaseJob dbjob = new SleepingDatabaseJob(name);
        try {
            dbjob.scheduleAndWaitForRunning();
            runnable.run();
        } finally {
            dbjob.disposeAndJoin();
        }
    }

    /**
     * @param name
     * @param runnable
     * @throws InterruptedException
     */
    public static void sleepUninterruptiblyWhile(String name, Runnable runnable) {
        SleepingDatabaseJob dbjob = new SleepingDatabaseJob(name);
        try {
            dbjob.scheduleAndWaitForRunningUninterruptibly();
            runnable.run();
        } finally {
            try {
                dbjob.disposeAndJoin();
            } catch (InterruptedException e) {
            }
        }
    }

}
