package org.simantics;

import java.util.Collection;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

/**
 * @author Antti Villberg
 */
public class DiskWarningSystem {

    public interface DiskWarningListener {
    	/*
    	 * @param available in bytes
    	 */
        void diskLow(long available);
    }

    private final Collection<DiskWarningListener> listeners = new CopyOnWriteArrayList<DiskWarningListener>();

    // In bytes
    private long limit = Long.MAX_VALUE;

    private boolean disposed = false;

    private ScheduledFuture<?> future;

    /*
     * @param limit in megabytes
     */
    public void setLimit(long limit) {
    	this.limit = limit*1024*1024;
    }

    public long get() {
    	return Simantics.getDiskBytes();
    }
    
    public boolean check() {
		return get() > limit;
    }
    	
    
    public DiskWarningSystem(int amount, TimeUnit unit) {
    	
    	future = Simantics.scheduleWithFixedDelay(new Runnable() {

    		@Override
    		public void run() {
    			if(!disposed && !check()) {
    				long bytes = get();
    				for (DiskWarningListener listener : listeners) {
    					listener.diskLow(bytes);
    				}
    			}
    		}

    	}, 0, amount, unit);
    	
    }

    public boolean addListener(DiskWarningListener listener) {
        return listeners.add(listener);
    }

    public boolean removeListener(DiskWarningListener listener) {
        return listeners.remove(listener);
    }

    public void dispose() {
        if (!disposed) {
            disposed = true;
            future.cancel(false);
            listeners.clear();
        }
    }

}