/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.simulation.experiment;

import java.util.UUID;
import java.util.concurrent.locks.Lock;

import org.eclipse.core.runtime.IProgressMonitor;
import org.simantics.databoard.accessor.Accessor;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.procedure.Listener;
import org.simantics.db.request.ExternalRead;
import org.simantics.utils.datastructures.ListenerList;

public abstract class Experiment implements IExperiment {

    protected Resource experiment;
    protected Resource model;
    protected ListenerList<IExperimentListener> listeners =
        new ListenerList<IExperimentListener>(IExperimentListener.class);
    protected ListenerList<IExperimentStatusListener> statusListeners =
        new ListenerList<IExperimentStatusListener>(IExperimentStatusListener.class);
    protected volatile ExperimentState state = ExperimentState.INITIALIZING;
    protected String identifier;

    public Experiment(Resource model) {
        this(null, model, UUID.randomUUID().toString());
    }

    public Experiment(Resource model, String identifier) {
        this(null, model, identifier);
    }

    public Experiment(Resource experiment, Resource model) {
        this(experiment, model, UUID.randomUUID().toString());
    }

    public Experiment(Resource experiment, Resource model, String identifier) {
        this.experiment = experiment;
        this.model = model;
        this.identifier = identifier;
    }
    
    @Override
    public <T> T getService(Class<T> clazz) {
    	return null;
    }

    @Override
    public ExperimentState getState() {
        return state;
    }
    
    @Override
    public ExperimentState getState(ReadGraph graph) throws DatabaseException {
    	return graph.syncRequest(EXPERIMENT_STATE_READ);
    }

    public void changeState(ExperimentState newState) {
        if (state == ExperimentState.DISPOSED) {
            if (newState != ExperimentState.DISPOSED) {
                // TODO: this might be worth logging as an error.
            }
            return;
        }
        if(newState != state) {
            state = newState;
            localStateChange();
            for(IExperimentListener listener : listeners.getListeners())
                listener.stateChanged(newState);
            EXPERIMENT_STATE_READ.run();
        }
    }

    protected void localStateChange() {
    }

    @Override
    public void addListener(IExperimentListener listener) {
        listeners.add(listener);
        listener.stateChanged(state);
    }

    @Override
    public Resource getResource() {
        return experiment;
    }

    @Override
    public Resource getModel() {
        return model;
    }

    @Override
    public void removeListener(IExperimentListener listener) {
        listeners.remove(listener);
    }

    @Override
    public void addStatusListener(IExperimentStatusListener listener) {
        statusListeners.add(listener);
    }

    @Override
    public void removeStatusListener(IExperimentStatusListener listener) {
        statusListeners.remove(listener);
    }

    @Override
    public void shutdown(IProgressMonitor monitor) {
        changeState(ExperimentState.DISPOSED);
    }

    @Override
    public Lock getDatasourceLock() {
        throw new UnsupportedOperationException("deprecated operation");
    }

    @Override
    public Accessor getAccessor() {
        return null;
    }

    @Override
    public String getIdentifier() {
        return identifier;
    }

    static class ExperimentStateRead implements ExternalRead<ExperimentState>, Runnable {
    	
    	final private Experiment experiment;
        private Listener<ExperimentState> listener = null;
        
        ExperimentStateRead(Experiment experiment) {
        	this.experiment = experiment;
        }

        @Override
        public void register(ReadGraph graph, final Listener<ExperimentState> procedure) {
            //System.out.println("IcTrackerRequest.register: " + procedure);
            listener = procedure;
            procedure.execute(experiment.state);
        }

        @Override
        public void unregistered() {
            //System.out.println("IcTrackerRequest.unregister: " + listener);
            listener = null;
        }

        @Override
        public void run() {
            Listener<ExperimentState> l = listener;
            //System.out.println("IcTrackerRequest.run: " + l);
            if (l != null)
                l.execute(experiment.state);
        }
    }

    private ExperimentStateRead EXPERIMENT_STATE_READ = new ExperimentStateRead(this);
    
}
