package org.simantics.scl.compiler.source.repository;

import java.util.ArrayList;
import java.util.Collection;

import org.simantics.scl.compiler.module.Module;
import org.simantics.scl.compiler.module.repository.UpdateListener;
import org.simantics.scl.compiler.module.repository.UpdateListener.Observable;
import org.simantics.scl.compiler.source.ModuleSource;
import org.simantics.scl.compiler.source.PrecompiledModuleSource;

import gnu.trove.map.hash.THashMap;
import gnu.trove.procedure.TObjectProcedure;

/**
 * An implementation of {@link ModuleSourceRepository} as a finite map.
 * This implementation does not support listening module changes,
 * so it should not be modified after it has been taken into use.
 * 
 * @author Hannu Niemist&ouml;
 */
public class MapModuleSourceRepository implements ModuleSourceRepository {
    THashMap<String, ModuleSource> modules = new THashMap<String, ModuleSource>();
    THashMap<String, String> documentations = new THashMap<String, String>();
    
    THashMap<String, ArrayList<UpdateListener>> listeners = new THashMap<String, ArrayList<UpdateListener>>(); 
    
    public MapModuleSourceRepository() {
    }
    
    public MapModuleSourceRepository(ModuleSource ... descriptors) {
        for(ModuleSource descriptor : descriptors)
            addModuleDescriptor(descriptor);
    }
    
    public MapModuleSourceRepository(Module ... modules) {
        for(Module module : modules)
            addModule(module);
    }
    
    public void addModuleDescriptor(ModuleSource descriptor) {
        modules.put(descriptor.getModuleName(), descriptor);
        synchronized (listeners) {
            ArrayList<UpdateListener> list = listeners.get(descriptor.getModuleName());
            if(list != null)
                for(UpdateListener listener : list.toArray(new UpdateListener[list.size()]))
                    listener.notifyAboutUpdate();
        }
    }
    
    public void addModule(Module module) {
        addModuleDescriptor(new PrecompiledModuleSource(module));
    }
    
    public void addDocumentation(String documentationName, String documentation) {
        documentations.put(documentationName, documentation);
    }

    @Override
    public ModuleSource getModuleSource(String moduleName,
            UpdateListener listener) {
        if(listener != null) {
            synchronized(listeners) {
                ArrayList<UpdateListener> list = listeners.get(moduleName);
                if(list == null) {
                    list = new ArrayList<UpdateListener>(2);
                    listeners.put(moduleName, list);
                }
                list.add(listener);
            }
            listener.addObservable(new Observable() {
                @Override
                public void removeListener(UpdateListener listener) {
                    synchronized(listeners) {
                        ArrayList<UpdateListener> list = listeners.get(moduleName);
                        if(list != null) {
                            list.remove(listener);
                            if(list.isEmpty())
                                listeners.remove(moduleName);
                        }
                    }
                }
            });
        }
        return modules.get(moduleName);
    }

    @Override
    public Collection<String> getModuleNames() {
        return modules.keySet();
    }
    
    @Override
    public void forAllModules(TObjectProcedure<String> procedure) {
        modules.forEachKey(procedure);
    }

    @Override
    public String getDocumentation(String documentationName) {
        return documentations.get(documentationName);
    }
    
    @Override
    public Collection<String> getDocumentationNames() {
        return documentations.keySet();
    }
}
