package org.simantics.scl.compiler.module.repository;

import gnu.trove.set.hash.THashSet;

/**
 * Listener that is notified about changes in modules and their dependencies. It is possible
 * to listen multiple different modules with one listener. When a change happens, the listener
 * automatically stops listening any other changes. The idea is that the client then asks all modules
 * again using the listener as a parameter.
 */
public abstract class UpdateListener {
    private final THashSet<Observable> observables = new THashSet<Observable>();  
    
    public interface Observable {
        void removeListener(UpdateListener listener);
    }
    
    public abstract void notifyAboutUpdate();
    
    /**
     * Registers an observable to the listener. The client code should never
     * call this method. It is needed so that it is possible to implement
     * {@link #stopListening}.
     */
    public void addObservable(Observable observable) {
        synchronized(observables) {
            observables.add(observable);
        }
    }

    /**
     * Stops listening changes. Returns true, if the listener was listening something. 
     */
    public boolean stopListening() {
        synchronized(observables) {
            if(observables.isEmpty())
                return false;
            for(Observable observable : observables)
                observable.removeListener(this);
            observables.clear();
            return true;
        }
    }
}
