package org.simantics.scl.compiler.internal.codegen.types;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;

import org.cojen.classfile.TypeDesc;
import org.simantics.scl.compiler.constants.generic.ClassRef;
import org.simantics.scl.runtime.function.Function;

public abstract class AbstractRuntimeJavaReferenceValidator 
implements JavaReferenceValidator<Class<?>, Method, Field, Constructor<?>> {

    @Override
    public abstract Class<?> findClass(TypeDesc name);
    
    @Override
    public boolean isInterface(Class<?> clazz) {
        return clazz.isInterface();
    }
    
    @Override
    public boolean isPublic(Class<?> clazz) {
        return Modifier.isPublic(clazz.getModifiers());
    }

    @Override
    public Method[] findCompatibleMethods(Class<?> clazz, boolean isStatic, String name, TypeDesc[] parameterTypes, TypeDesc returnType) {
        Class<?>[] parameterClasses = new Class[parameterTypes.length];
        for(int i=0;i<parameterTypes.length;++i)
            parameterClasses[i] = findClass(parameterTypes[i]);
        Class<?> returnClass = findClass(returnType);
        
        ArrayList<Method> methods = new ArrayList<Method>(2);
        
        methodLoop: 
        for(Method method : clazz.getMethods()) {
            if(!method.getName().equals(name))
                continue;
            if(Modifier.isStatic(method.getModifiers()) != isStatic)
                continue;
            
            Class<?>[] parameters = method.getParameterTypes();
            if(parameters.length != parameterClasses.length)
                continue;
            for(int i=0;i<parameters.length;++i)
                if(!parameters[i].isAssignableFrom(parameterClasses[i]))
                    continue methodLoop;

            if(!returnClass.isAssignableFrom(method.getReturnType()))
                continue;            
            
            methods.add(method);
        }       
        
        return methods.toArray(new Method[methods.size()]);
    }

    @Override
    public TypeDesc getReturnType(Method method) {
        return TypeDesc.forClass(method.getReturnType());
    }

    @Override
    public TypeDesc[] getParameterTypes(Method method) {
        Class<?>[] parameters = method.getParameterTypes();
        TypeDesc[] result = new TypeDesc[parameters.length];
        for(int i=0;i<parameters.length;++i)
            result[i] = TypeDesc.forClass(parameters[i]);
        return result;
    }

    @Override
    public Constructor<?>[] findCompatibleConstructors(Class<?> clazz,
            TypeDesc[] types) {
        Class<?>[] classes = new Class[types.length];
        for(int i=0;i<types.length;++i)
            classes[i] = findClass(types[i]);
        int maxArity = types.length-1;
        
        ArrayList<Constructor<?>> constructors = new ArrayList<Constructor<?>>(2);
        
        methodLoop: 
        for(Constructor<?> constructor : clazz.getConstructors()) {           
            Class<?>[] parameters = constructor.getParameterTypes();
            int arity = parameters.length;
            if(arity > maxArity)
                continue;
            for(int i=0;i<parameters.length;++i)
                if(!parameters[i].isAssignableFrom(classes[i]))
                    continue methodLoop;
            
            if(arity == maxArity) {
                if(!classes[maxArity].isAssignableFrom(clazz))
                    continue;
            }
            else {
                if(!Function.class.isAssignableFrom(clazz))
                    continue;
            }
            
            constructors.add(constructor);
        }       
        
        return constructors.toArray(new Constructor[constructors.size()]);
    }

    @Override
    public TypeDesc[] getConstructorParameterTypes(Constructor<?> constructor) {
        Class<?>[] parameters = constructor.getParameterTypes();
        TypeDesc[] result = new TypeDesc[parameters.length];
        for(int i=0;i<parameters.length;++i)
            result[i] = TypeDesc.forClass(parameters[i]);
        return result;
    }

    @Override
    public Field findField(Class<?> clazz, String name) {
        try {
            return clazz.getField(name);
        } catch(NoSuchFieldException e) {
            return null;
        }
    }

    @Override
    public boolean isStaticField(Field field) {
        return Modifier.isStatic(field.getModifiers());
    }

    @Override
    public TypeDesc getFieldType(Field field) {
        return TypeDesc.forClass(field.getType());
    }

    @Override
    public boolean isAssignableFrom(TypeDesc to, TypeDesc from) {
        if(to.equals(from) || to.equals(TypeDesc.OBJECT))
            return true;
        Class<?> toClass = findClass(to);
        Class<?> fromClass = findClass(from);
        if(toClass == null || fromClass == null)
            // Note: I added this branch when I noticed that type can be
            //       one from the module under compilation.
            // Note2: A problem with this seems to be that also 
            //        some other type descs go to null, such as double[][]
            return false; 
        else
            return toClass.isAssignableFrom(fromClass);
    }

    @Override
    public Method[] chooseBest(Method[] methods) {
        ArrayList<Method> newResult = new ArrayList<Method>();
        for(Method method : methods)
            if(!method.isSynthetic())
                newResult.add(method);
        return newResult.toArray(new Method[newResult.size()]);
    }
    
    @Override
    public ClassRef getClassRef(String className) {
        Class<?> clazz = findClass(TypeDesc.forClass(className));
        if(clazz == null)
            return null;
        return new ClassRef(clazz);
    }

}
