package org.simantics.scl.compiler.elaboration.expressions;

import org.simantics.scl.compiler.common.exceptions.InternalCompilerError;
import org.simantics.scl.compiler.compilation.CompilationContext;
import org.simantics.scl.compiler.elaboration.contexts.SimplificationContext;
import org.simantics.scl.compiler.elaboration.contexts.TranslationContext;
import org.simantics.scl.compiler.elaboration.contexts.TypingContext;
import org.simantics.scl.compiler.errors.Locations;
import org.simantics.scl.compiler.internal.codegen.references.IVal;
import org.simantics.scl.compiler.internal.codegen.writer.CodeWriter;
import org.simantics.scl.compiler.types.Types;
import org.simantics.scl.compiler.types.exceptions.MatchException;
import org.simantics.scl.compiler.types.util.MultiFunction;

public class EViewPattern extends Expression {
    public Expression expression;
    public Expression pattern;
    
    public EViewPattern(Expression expression, Expression pattern) {
        this.expression = expression;
        this.pattern = pattern;
    }
    
    @Override
    public Expression inferType(TypingContext context) {
        context.setInPattern(false);
        expression = expression.inferType(context);
        context.setInPattern(true);
        MultiFunction mfun;
        try {
            mfun = Types.matchFunction(expression.getType(), 1);
        } catch (MatchException e) {
            context.getErrorLog().log(expression.location, "Expected a function as a transformation expression.");
            return new EError(location);
        }
        setType(mfun.parameterTypes[0]);
        pattern.checkType(context, mfun.returnType);
        return this;
    }

    @Override
    protected void updateType() throws MatchException {
        MultiFunction mfun = Types.matchFunction(expression.getType(), 1);
        setType(mfun.parameterTypes[0]);
    }

    @Override
    public IVal toVal(CompilationContext context, CodeWriter w) {
        throw new InternalCompilerError(location, "EViewPattern.toVal should not be invoked.");
    }

    @Override
    public Expression resolve(TranslationContext context) {
        context.getErrorLog().log("View pattern cannot occur only in patterns. Maybe you are missing '\\' in front of a lambda experssion?");
        return new EError(location);
    }

    @Override
    public Expression resolveAsPattern(TranslationContext context) {
        expression = expression.resolve(context);
        pattern = pattern.resolveAsPattern(context);
        return this;
    }
    
    @Override
    public void setLocationDeep(long loc) {
        if(location == Locations.NO_LOCATION) {
            location = loc; 
            expression.setLocationDeep(loc);
            pattern.setLocationDeep(loc);
        }
    }

    @Override
    public void accept(ExpressionVisitor visitor) {
        visitor.visit(this);
    }

    @Override
    public Expression accept(ExpressionTransformer transformer) {
        return transformer.transform(this);
    }
    
    @Override
    public Expression simplify(SimplificationContext context) {
        expression = expression.simplify(context);
        pattern = pattern.simplify(context);
        return this;
    }

}
