package org.simantics.scl.compiler.elaboration.expressions;

import org.simantics.scl.compiler.common.exceptions.InternalCompilerError;
import org.simantics.scl.compiler.compilation.CompilationContext;
import org.simantics.scl.compiler.elaboration.contexts.ReplaceContext;
import org.simantics.scl.compiler.elaboration.contexts.SimplificationContext;
import org.simantics.scl.compiler.elaboration.contexts.TranslationContext;
import org.simantics.scl.compiler.elaboration.contexts.TypingContext;
import org.simantics.scl.compiler.errors.Locations;
import org.simantics.scl.compiler.internal.codegen.references.IVal;
import org.simantics.scl.compiler.internal.codegen.writer.CodeWriter;
import org.simantics.scl.compiler.internal.interpreted.IExpression;
import org.simantics.scl.compiler.top.ExpressionInterpretationContext;
import org.simantics.scl.compiler.types.TVar;
import org.simantics.scl.compiler.types.Types;
import org.simantics.scl.compiler.types.exceptions.MatchException;

public class ELambdaType extends Expression {
    public TVar[] parameters;
    public Expression value;
    
    public ELambdaType(TVar[] parameters, Expression value) {
        super(value.getLocation());
        this.parameters = parameters;
        this.value = value;
    }

    @Override
    protected void updateType() throws MatchException {
        setType(Types.forAll(parameters, value.getType()));
    }

    @Override
    public IVal toVal(CompilationContext context, CodeWriter w) {
        return lambdaToVal(context, w);
    }

    @Override
    public Expression simplify(SimplificationContext context) {
        value = value.simplify(context);
        return this;
    }

    @Override
    public Expression resolve(TranslationContext context) {
        value = value.resolve(context);
        return this;
    }

    @Override
    public Expression replace(ReplaceContext context) {
        TVar[] newParameters = new TVar[parameters.length];
        for(int i=0;i<parameters.length;++i) {
            TVar var = Types.var(parameters[i].getKind());
            newParameters[i] = var;
            context.tvarMap.put(parameters[i], var);
        }
            
        ELambdaType result = new ELambdaType(newParameters, value.replace(context));
        for(int i=0;i<parameters.length;++i)
            context.tvarMap.remove(parameters[i]);
        return result;
    }
    
    @Override
    public IExpression toIExpression(ExpressionInterpretationContext target) {
        return value.toIExpression(target);
    }
    
    @Override
    public Expression inferType(TypingContext context) {
        throw new InternalCompilerError("Should not type check " + getClass().getSimpleName() + ".");
    }
    
    @Override
    public void setLocationDeep(long loc) {
        if(location == Locations.NO_LOCATION) {
            location = loc;
            value.setLocationDeep(loc);
        }
    }
    
    @Override
    public void accept(ExpressionVisitor visitor) {
        visitor.visit(this);
    }
    
    @Override
    public Expression accept(ExpressionTransformer transformer) {
        return transformer.transform(this);
    }
    
    @Override
    public int getSyntacticFunctionArity() {
        return value.getSyntacticFunctionArity();
    }

}
