package org.simantics.scl.compiler.commands;

import java.io.BufferedReader;
import java.io.IOException;

import org.junit.Assert;
import org.simantics.scl.runtime.reporting.AbstractSCLReportingHandler;
import org.simantics.scl.runtime.reporting.SCLReportingHandler;

public class TestScriptExecutor {

    private final CommandSession session;
    private final BufferedReader reader;
    private final SCLReportingHandler handler;
    private boolean ignorePrints;
    
    public TestScriptExecutor(CommandSession session, BufferedReader reader, SCLReportingHandler handler) {
        this(session, reader, handler, false);
    }
    
    public TestScriptExecutor(CommandSession session, BufferedReader reader, SCLReportingHandler handler, boolean ignorePrints) {
        this.session = session;
        this.reader = reader;
        this.handler = handler == null ? SCLReportingHandler.DEFAULT : handler;
        this.ignorePrints = ignorePrints;
    }

    public void execute() throws IOException {
        StringBuilder command = new StringBuilder();
        StringBuilder response = new StringBuilder();
        while(true) {
            // Read command
            while(true) {
                String line = reader.readLine();
                if(line == null) {
                    execute(command.toString(), response.toString());
                    return;
                }
                else if(line.startsWith("> "))
                    command.append(line.substring(2)).append('\n');
                else {
                    response.append(line).append('\n');
                    break;
                }
            }
            
            // Read response
            while(true) {
                String line = reader.readLine();
                if(line == null) {
                    execute(command.toString(), response.toString());
                    return;
                }
                else if(line.startsWith("> ")) {
                    execute(command.toString(), response.toString());
                    command = new StringBuilder();
                    response = new StringBuilder();
                    command.append(line.substring(2)).append('\n');
                    break;
                }
                else
                    response.append(line).append('\n');
            }
        }
    }

    private void execute(String command, String expectedResponse) {
        final StringBuilder actualResponse = new StringBuilder();
        final StringBuilder errors = new StringBuilder();
        session.execute(command,
                new AbstractSCLReportingHandler() {
                    
                    @Override
                    public void print(String text) {
                        handler.print(text);
                        if (!ignorePrints)
                            actualResponse.append(text).append('\n');
                    }
                    
                    @Override
                    public void printError(String error) {
                        handler.printError(error);
                        errors.append(error).append('\n');
                    }
                    
                    @Override
                    public void printCommand(String command) {
                        handler.printCommand(command);
                    }
                });
        if(errors.length() > 0)
            Assert.fail("Command '" + command.trim() + "' produced error " + errors.toString().trim() + ".");
        String expected = expectedResponse.trim().replaceAll("\\r", "");
        String actual = actualResponse.toString().trim().replaceAll("\\r", "");
        Assert.assertEquals(expected, actual);
    }
    
}
