/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.scenegraph.utils;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.RenderingHints;
import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;
import java.util.Objects;

import org.simantics.scenegraph.g2d.G2DRenderingHints;
import org.simantics.scenegraph.g2d.color.ColorFilter;
import org.simantics.scenegraph.g2d.color.Graphics2DWithColorFilter;

import com.kitfox.svg.SVGDiagram;
import com.kitfox.svg.SVGException;

/**
 * @author J-P Laine
 */
public class BufferedImage  {
    SVGDiagram                 source;
    Rectangle2D                imageBounds;
    Point                      referenceSize;

    java.awt.image.BufferedImage   buffer = null;
    AffineTransform previousTransform = null;
    ColorFilter previousColorFilter = null;

    /**
     * @param original
     * @param imageBounds
     * @param referenceSize a reference size for the rasterized images or
     *        <code>null</code> to not specify one in which case a default
     *        resolution is used
     */
    public BufferedImage(SVGDiagram original, Rectangle2D imageBounds, Point referenceSize) {
        this.source = original;
        this.imageBounds = imageBounds;
        this.referenceSize = referenceSize;
    }

    public BufferedImage(SVGDiagram original) {
        this.source = original;
        this.imageBounds = original.getViewRect();
        this.referenceSize = null;
    }

    public java.awt.image.BufferedImage getBuffer() {
    	return buffer;
    }
    
    @Override
    public void finalize() throws Throwable {
        //System.out.println("BufferedImage.finalize()");
        releaseRaster();
        super.finalize();
    }

    public synchronized void releaseRaster() {
        source = null;
        if (buffer != null) {
            buffer.flush();
            buffer = null;
        }
    }

    public void paintToBuffer(AffineTransform transform, float margin) throws SVGException {
        paintToBuffer(transform, margin);
    }

    public void paintToBuffer(AffineTransform transform, ColorFilter colorFilter, float margin) throws SVGException {

        int w = (int)((imageBounds.getWidth()+margin*2) * transform.getScaleX());
        int h = (int)((imageBounds.getHeight()+margin*2) * transform.getScaleY());
        if(w < 1) w = 1;
        if(h < 1) h = 1; // FIXME
    	
        buffer = new java.awt.image.BufferedImage(w, h, java.awt.image.BufferedImage.TYPE_INT_ARGB);
    	
    	Graphics2D b2g = (Graphics2D)buffer.getGraphics();

        b2g.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_SPEED);
        b2g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
        b2g.setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BILINEAR);

        b2g.setBackground(Color.WHITE);
        b2g.clearRect(0, 0, w, h);
        b2g.translate(margin, margin);
        b2g.scale(transform.getScaleX(), transform.getScaleY());
        b2g.translate(-imageBounds.getMinX(), -imageBounds.getMinY());
        if (colorFilter != null) {
            source.render(new Graphics2DWithColorFilter(b2g, colorFilter));
        } else {
            source.render(b2g);
        }

    }
    
    public void paint(Graphics2D g) {
        float margin = 5;

        ColorFilter colorFilter = (ColorFilter) g.getRenderingHint(G2DRenderingHints.KEY_COLOR_FILTER);
        if(previousTransform == null || previousTransform.getScaleX() != g.getTransform().getScaleX() || previousTransform.getScaleY() != g.getTransform().getScaleY() || !Objects.equals(colorFilter, previousColorFilter)) {
//        	setupSourceRender(g);
            try {
                previousColorFilter = colorFilter;
                previousTransform = (AffineTransform)g.getTransform().clone();
                paintToBuffer(previousTransform, previousColorFilter, margin);
            } catch (SVGException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }
        }

        g.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_SPEED);
        g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_OFF);
        g.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_OFF);
        g.setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_NEAREST_NEIGHBOR);

        AffineTransform at = (AffineTransform)g.getTransform().clone();
        try {
            g.translate(imageBounds.getMinX(), imageBounds.getMinY());
            g.scale(1/previousTransform.getScaleX(), 1/previousTransform.getScaleY());
            g.translate(-margin, -margin);

            g.drawImage(buffer, 0, 0, null);
        } finally {
            g.setTransform(at);
        }
    }
}
