/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.scenegraph.g2d.nodes;

import java.awt.AlphaComposite;
import java.awt.Composite;
import java.awt.Graphics2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.Map;

import org.simantics.scenegraph.INode;
import org.simantics.scenegraph.g2d.G2DRenderingHints;
import org.simantics.scenegraph.g2d.IG2DNode;
import org.simantics.scenegraph.g2d.color.ColorFilter;
import org.simantics.scenegraph.g2d.color.ColorFilterStack;
import org.simantics.scenegraph.g2d.color.Graphics2DWithColorFilter;
import org.simantics.scenegraph.g2d.events.EventTypes;
import org.simantics.scenegraph.g2d.events.MouseEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseDragBegin;
import org.simantics.scenegraph.utils.InitValueSupport;
import org.simantics.scenegraph.utils.NodeUtil;

public class SingleElementNode extends TransformNode implements InitValueSupport {

    private static final long serialVersionUID = -4982578347433716440L;

    private transient TransferableProvider transferableProvider;
    protected Composite composite;
    private ColorFilter colorFilter;
    private ColorFilter hoverFilter;
    protected Boolean visible = Boolean.TRUE;
    protected Boolean hidden = Boolean.FALSE;
    private transient Object key;
    private transient String typeClass;
    private transient Map<String, Object> parameters;

    public void setKey(Object key) {
        this.key = key;
    }

    public void setTypeClass(String typeClass) {
        this.typeClass = typeClass;
    }

    public Object getKey() {
        return key;
    }

    public String getTypeClass() {
        return typeClass;
    }

    public void setParameters(Map<String,Object> parameters) {
        this.parameters = parameters;
    }

    public <T> T getParameter(String key) {
        if (parameters != null) {
            @SuppressWarnings("unchecked")
            T t = (T) parameters.get(key);
            if(t != null) return t;
        }
        INode parent = NodeUtil.getNearestParentOfType(this, SingleElementNode.class);
        if (parent instanceof SingleElementNode) {
            return ((SingleElementNode)parent).getParameter(key);
        }
        return null;
    }

    public void setTransferableProvider(TransferableProvider transferableProvider) {
        if (transferableProvider != this.transferableProvider) {
            if (this.transferableProvider != null)
                removeEventHandler(this);
            if (transferableProvider != null)
                addEventHandler(this);
            this.transferableProvider = transferableProvider;
        }
    }

    @Override
    public boolean validate() {
        return visible && !hidden;
    }

    @SyncField("composite")
    public void setComposite(Composite composite) {
        this.composite = composite;
    }
    
    public void setColorFilter(ColorFilter colorFilter) {
        this.colorFilter = colorFilter;
    }

    public ColorFilter getColorFilter() {
        return colorFilter;
    }

    public void setHoverFilter(ColorFilter hoverFilter) {
        this.hoverFilter = hoverFilter;
    }

    public ColorFilter getHoverFilter() {
        return hoverFilter;
    }

    @SyncField("visible")
    public void setVisible(Boolean visible) {
        this.visible = visible;
    }

    public boolean isVisible() {
        return visible;
    }

    @SyncField("hidden")
    public void setHidden(Boolean hidden) {
        this.hidden = hidden;
    }

    public boolean isHidden() {
        return hidden;
    }

    @Override
    public void render(Graphics2D g2d) {
        if (!visible || hidden)
            return;

        Graphics2D g;

        ColorFilter currentFilter = null;
        if (colorFilter != null && hoverFilter != null) {
            ColorFilterStack stack = new ColorFilterStack();
            stack.addColorFilter(colorFilter);
            currentFilter = stack;
        } else if (colorFilter != null) {
            currentFilter = colorFilter;
        } else if (hoverFilter != null) {
            currentFilter = hoverFilter; 
        }
        
        if (currentFilter != null) {
            g = new Graphics2DWithColorFilter(g2d, currentFilter);
            g.setRenderingHint(G2DRenderingHints.KEY_COLOR_FILTER, currentFilter);
        } else {
            g = g2d;
        }

        Composite oldComposite = null;
        if(composite != null) {
            oldComposite = g.getComposite();
            g.setComposite(composite);
        }
        if(alphaComposite != null) {
            if (oldComposite == null)
                oldComposite = g.getComposite();
            g.setComposite(alphaComposite);
        }

        super.render(g);

        if (oldComposite != null)
            g.setComposite(oldComposite);

        if (currentFilter != null) {
            g.setRenderingHint(G2DRenderingHints.KEY_COLOR_FILTER, null);
        }
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(super.toString());
        sb.append(" [");
        if (composite != null) {
            sb.append("composite=(");
            AlphaComposite ac = (AlphaComposite) composite;
            sb.append(ac.getRule());
            sb.append(", ");
            sb.append(ac.getAlpha());
            sb.append("), ");
        }
        sb.append("visible=");
        sb.append(visible);
        sb.append("]");
        return sb.toString();
    }

    protected Composite alphaComposite = null;

    @PropertySetter("alpha")
    @SyncField("alphaComposite")
    public void setAlphaComposite(Composite alphaComposite) {
        this.alphaComposite = alphaComposite;
    }

    @Override
    public void initValues() {
        alphaComposite = null;
        for (IG2DNode node : getSortedNodes()) {
            if (node instanceof InitValueSupport) {
                ((InitValueSupport) node).initValues();
            }
        }
    }

    @Override
    public void cleanup() {
        if (transferableProvider != null) {
            removeEventHandler(this);
            transferableProvider = null;
        }
        super.cleanup();
    }

    protected boolean isDragStartEvent(MouseEvent e) {
        return e.isControlDown();
    }

    protected boolean hitTest(MouseEvent event) {
        Rectangle2D bounds = super.getBoundsInLocal(true);
        if (bounds == null)
            return false;
        Point2D localPos = NodeUtil.worldToLocal(this, event.controlPosition, new Point2D.Double());
        double x = localPos.getX();
        double y = localPos.getY();
        boolean hit = bounds.contains(x, y);
        return hit;
    }

    @Override
    protected boolean mouseDragged(MouseDragBegin e) {
        if (transferableProvider == null
                || !isDragStartEvent(e)
                || !hitTest(e))
            return false;
        e.transferable = transferableProvider.create();
        return e.transferable != null;
    }

    @Override
    public int getEventMask() {
        return EventTypes.MouseDragBeginMask;
    }

    public void beforeRender(Graphics2D g) {
        g.setRenderingHint(G2DRenderingHints.KEY_BEGIN_ELEMENT, "definedElement");
    }

    public void afterRender(Graphics2D g) {
        g.setRenderingHint(G2DRenderingHints.KEY_END_ELEMENT, "definedElement");
    }

}
