/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.scenegraph.g2d.nodes;

import java.awt.Color;
import java.awt.Composite;
import java.awt.Graphics2D;
import java.awt.Stroke;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;

import org.simantics.diagram.connection.RouteGraph;
import org.simantics.scenegraph.INode;
import org.simantics.scenegraph.g2d.G2DRenderingHints;
import org.simantics.scenegraph.g2d.IG2DNode;
import org.simantics.scenegraph.g2d.events.MouseEvent;
import org.simantics.scenegraph.g2d.nodes.connection.RouteGraphNode;
import org.simantics.scenegraph.utils.InitValueSupport;
import org.simantics.scenegraph.utils.NodeUtil;
import org.simantics.scenegraph.utils.NodeUtil.NodeProcedure;

/**
 * @author Tuukka Lehtonen
 */
public class ConnectionNode extends SingleElementNode implements InitValueSupport {

    private static final long serialVersionUID = 8657421058765967521L;

    private static class SetAlphaCompositeProcedure implements NodeProcedure<Object> {
        public Composite composite;
        @Override
        public Object execute(INode node, String id) {
            if (node instanceof EdgeNode)
                ((EdgeNode) node).setAlphaComposite(composite);
//          NodeUtil.setPropertyIfSupported("alpha", alphaComposite, edge);
            return null;
        }
    };

    private static class SetDynamicStrokeProcedure implements NodeProcedure<Object> {
        public Stroke stroke;
        @Override
        public Object execute(INode node, String id) {
            if (node instanceof EdgeNode)
                ((EdgeNode) node).setDynamicStroke(stroke);
            else if (node instanceof RouteGraphNode)
                ((RouteGraphNode) node).setDynamicStroke(stroke);
//          NodeUtil.setPropertyIfSupported("width", stroke, edge);
            return null;
        }
    };

    private static class SetDynamicColorProcedure implements NodeProcedure<Object> {
        public Color color;
        @Override
        public Object execute(INode node, String id) {
            if (node instanceof EdgeNode)
                ((EdgeNode) node).setDynamicColor(color);
            else if (node instanceof RouteGraphNode)
                ((RouteGraphNode) node).setDynamicColor(color);
//          NodeUtil.setPropertyIfSupported("color", color, edge);
            return null;
        }
    };

//    private static class SetValueProcedure implements NodeProcedure<Object> {
//        public String key;
//        public Object value;
//        @Override
//        public Object execute(INode node, String id) {
//            NodeUtil.setPropertyIfSupported(key, value, node);
////            edge.setValue(key, value);
//            return null;
//        }
//    };

    private static final SetAlphaCompositeProcedure setAlphaCompositeProcedure = new SetAlphaCompositeProcedure();
    private static final SetDynamicStrokeProcedure  setDynamicStrokeProcedure  = new SetDynamicStrokeProcedure();
    private static final SetDynamicColorProcedure   setDynamicColorProcedure   = new SetDynamicColorProcedure();

    private static NodeProcedure<?> initValuesProcedure = new NodeProcedure<Object>() {
        @Override
        public Object execute(INode node, String id) {
            if (node instanceof InitValueSupport)
                ((InitValueSupport) node).initValues();
            return null;
        }
    };

    @PropertySetter("alpha")
    public void setAlphaComposite(Composite alphaComposite) {
        setAlphaCompositeProcedure.composite = alphaComposite;
        forEdges(this, setAlphaCompositeProcedure);
    }
    
    @PropertySetter("width")
    public void setDynamicStroke(Stroke stroke) {
        setDynamicStrokeProcedure.stroke = stroke;
        forEdges(this, setDynamicStrokeProcedure);
    }

    @PropertySetter("color")
    public void setDynamicColor(Color color) {
        setDynamicColorProcedure.color = color;
        forEdges(this, setDynamicColorProcedure);
    }

    @Override
    public void initValues() {
        NodeUtil.forChildren(this, initValuesProcedure, null);
    }

    private void forEdges(SingleElementNode node, NodeProcedure<?> procedure) {
        for (String childId : node.getSortedNodesById()) {
            IG2DNode child = node.getNode(childId);
            if (child instanceof SingleElementNode) {
                forEdges((SingleElementNode) child, procedure);
            } else if (child instanceof EdgeNode) {
                procedure.execute(child, childId);
            } else if (child instanceof RouteGraphNode) {
                procedure.execute(child, childId);
            }
        }
    }

    @Override
    protected boolean hitTest(MouseEvent event) {
        if (!super.hitTest(event))
            return false;

        Point2D pos = NodeUtil.worldToLocal(this, event.controlPosition, new Point2D.Double());
        double tolerance = 0.5;
        // TODO: change tolerance based on zoom level

        for (IG2DNode child : getSortedNodes()) {
            if (child instanceof RouteGraphNode) {
                RouteGraphNode rgn = (RouteGraphNode) child;
                RouteGraph rg = rgn.getRouteGraph();
                Object pick = rg.pickLine(pos.getX(), pos.getY(), tolerance);
                if (pick != null)
                    return true;
            }
        }
        return false;
    }

    @Override
    public void beforeRender(Graphics2D g) {
        g.setRenderingHint(G2DRenderingHints.KEY_BEGIN_ELEMENT, "connection");
    }
    
    @Override
    public void afterRender(Graphics2D g) {
        g.setRenderingHint(G2DRenderingHints.KEY_END_ELEMENT, "connection");
    }

    @Override
    public Rectangle2D getBoundsInLocal() {
        // #134: Route graph connections render their own selection.
        // ElementPainter will place an empty G2DParentNode
        // called "selection" under this ConnectionNode which
        // should be ignored in bounds calculations.
        // Otherwise this node will not support being inserted
        // into a spatial search structure and further selections
        // will fail.
        return super.getBoundsInLocal(true);
    }

}
