/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.scenegraph.g2d.events.command;

import java.awt.event.KeyEvent;
import java.util.Arrays;

/**
 * Describes a key binding of a command
 * 
 * @See {@link Command} Command class
 * @See {@link Commands} Default commands
 * @See KeyToCommand Participant that forwards key presses to commands
 * 
 * @author Toni Kalajainen
 */
public class CommandKeyBinding {

	public static final CommandKeyBinding[] DEFAULT_BINDINGS = {
		new CommandKeyBinding(Commands.CANCEL, "Press Escape to cancel the current operation.", KeyEvent.VK_ESCAPE),
		new CommandKeyBinding(Commands.PAN_LEFT, null, KeyEvent.VK_LEFT),
		new CommandKeyBinding(Commands.PAN_RIGHT, null, KeyEvent.VK_RIGHT),
		new CommandKeyBinding(Commands.PAN_UP, null, KeyEvent.VK_UP),
		new CommandKeyBinding(Commands.PAN_DOWN, null, KeyEvent.VK_DOWN),
//		new CommandKeyBinding(Commands.ZOOM_TO_FIT, null, KeyEvent.VK_1),
//        new CommandKeyBinding(Commands.ZOOM_TO_FIT_HORIZ, null, KeyEvent.VK_2),
//        new CommandKeyBinding(Commands.ZOOM_TO_FIT_VERT, null, KeyEvent.VK_3),
//        new CommandKeyBinding(Commands.AUTOSCALE, null, KeyEvent.VK_4),
        new CommandKeyBinding(Commands.ZOOM_TO_SELECTION, null, KeyEvent.VK_2),
        new CommandKeyBinding(Commands.ZOOM_TO_PAGE, null, KeyEvent.VK_3),
		new CommandKeyBinding(Commands.ZOOM_IN, null, KeyEvent.VK_PLUS),
		new CommandKeyBinding(Commands.ZOOM_IN, null, KeyEvent.VK_ADD),
		new CommandKeyBinding(Commands.ZOOM_OUT, null, KeyEvent.VK_MINUS),
		new CommandKeyBinding(Commands.ZOOM_OUT, null, KeyEvent.VK_SUBTRACT),
		new CommandKeyBinding(Commands.ZOOM_TO_AREA, null, KeyEvent.VK_F3),
		new CommandKeyBinding(Commands.REFRESH, null, KeyEvent.VK_F5),
		new CommandKeyBinding(Commands.RENAME, null, KeyEvent.VK_F2),
		new CommandKeyBinding(Commands.DELETE, null, KeyEvent.VK_DELETE),
		new CommandKeyBinding(Commands.SELECT_ALL, null, -KeyEvent.VK_A, KeyEvent.VK_CONTROL),
		new CommandKeyBinding(Commands.SELECT_ALL, null, KeyEvent.VK_A, KeyEvent.VK_CONTROL),
		new CommandKeyBinding(Commands.CUT, null, KeyEvent.VK_X, KeyEvent.VK_CONTROL),
		new CommandKeyBinding(Commands.COPY, null, KeyEvent.VK_C, KeyEvent.VK_CONTROL),
		new CommandKeyBinding(Commands.PASTE, null, KeyEvent.VK_V, KeyEvent.VK_CONTROL),
		new CommandKeyBinding(Commands.PRINT, null, KeyEvent.VK_P, KeyEvent.VK_CONTROL),
		new CommandKeyBinding(Commands.PDFPRINT, null, KeyEvent.VK_P, KeyEvent.VK_ALT),
		new CommandKeyBinding(Commands.BRING_UP, null, KeyEvent.VK_PAGE_UP),
		new CommandKeyBinding(Commands.SEND_DOWN, null, KeyEvent.VK_PAGE_DOWN),		
		new CommandKeyBinding(Commands.BRING_TO_TOP, null, KeyEvent.VK_HOME),
		new CommandKeyBinding(Commands.SEND_TO_BOTTOM, null, KeyEvent.VK_END),
		new CommandKeyBinding(Commands.ROTATE_ELEMENT_CW, null, KeyEvent.VK_PERIOD),
		new CommandKeyBinding(Commands.ROTATE_ELEMENT_CCW, null, KeyEvent.VK_COMMA),
		new CommandKeyBinding(Commands.FLIP_ELEMENT_HORIZONTAL, null, KeyEvent.VK_F, -KeyEvent.VK_CONTROL),
		new CommandKeyBinding(Commands.FLIP_ELEMENT_VERTICAL, null, KeyEvent.VK_V, -KeyEvent.VK_CONTROL),
//		new CommandKeyBinding(Commands.ROTATE_CANVAS_CW_GRAB, null, KeyEvent.VK_PERIOD),
//		new CommandKeyBinding(Commands.ROTATE_CANVAS_CW_RELEASE, null, -KeyEvent.VK_PERIOD),
//		new CommandKeyBinding(Commands.ROTATE_CANVAS_CCW_GRAB, null, KeyEvent.VK_COMMA),
//		new CommandKeyBinding(Commands.ROTATE_CANVAS_CCW_RELEASE, null, -KeyEvent.VK_COMMA),
		new CommandKeyBinding(Commands.GRID_TOGGLE, null, KeyEvent.VK_G),
		new CommandKeyBinding(Commands.RULER_TOGGLE, null, KeyEvent.VK_R),
		new CommandKeyBinding(Commands.FOCUS_TOOLTIP, null, KeyEvent.VK_F9)
		};
	
	/** 
	 * Required keys including modifiers, virtual codes.
	 * The code at first index is the trigger code, and other codes are 
	 * modifiers (ctrl, shift, etc).
	 * Negative code is release (index=0) or modifier is up (index>0) 
	 */
	public final int [] keyCode;
	/** Command to trigger */
	public final Command command;
	/** Guide shown in UI about how to use the binding */
	public final String guide;
	
	private int hash;
	
	public CommandKeyBinding(Command command, String guide, int...keyCode) {
		assert(keyCode.length>0);
		this.command = command;
		this.keyCode = keyCode;
		this.guide = guide;
		this.hash = Arrays.hashCode(keyCode) + 13*command.hashCode();
	}
		
	
	@Override
	public int hashCode() {
		return hash;
	}
	
	@Override
	public boolean equals(Object obj) {
		if (!(obj instanceof CommandKeyBinding)) return false;
		CommandKeyBinding other = (CommandKeyBinding) obj;		
		return other.command.equals(command) && Arrays.equals(keyCode, other.keyCode);
	}
	
}
