/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.scenegraph.g2d.events.adapter;

import java.awt.AWTEvent;
import java.awt.event.AWTEventListener;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;
import java.awt.geom.Point2D;
import org.simantics.scenegraph.g2d.events.IEventHandler;
import org.simantics.scenegraph.g2d.events.IEventQueue;
import org.simantics.scenegraph.g2d.events.KeyEvent.KeyPressedEvent;
import org.simantics.scenegraph.g2d.events.KeyEvent.KeyReleasedEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseButtonPressedEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseButtonReleasedEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseDoubleClickedEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseEnterEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseExitEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseMovedEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseWheelMovedEvent;
import org.simantics.scenegraph.g2d.events.adapter.AbstractEventAdapter;


/**
 * Listens to objectinput stream and reads AWT events
 * Adapts AWT events into G2D events.
 * 
 * Sends adapted events to IEventHandler
 */
public class AWTRemoteEventAdapter extends AbstractEventAdapter implements AWTEventListener, MouseListener, MouseMotionListener, MouseWheelListener, KeyListener {

    /** Mouse id of the default mouse */
    public static final int MOUSE_ID     = 0;

    private long[]          pressTime    = new long[5];
    int                     buttonStatus = 0;

    /**
     * create new adapter
     * @param input the input stream where the events are read from
     * @param sender the sender field in the events
     * @param delegator the target of the adapted events
     */
    public AWTRemoteEventAdapter(Object sender, IEventHandler delegator) {
        super(sender, delegator);
    }

    /**
     * create new adapter
     * @param sender the sender field in the events
     * @param queue
     */
    public AWTRemoteEventAdapter(Object sender, IEventQueue queue) {
        super(sender, queue);
    }	

    private Point2D getControlPosition(MouseEvent e)
    {
        return new Point2D.Double(e.getX(), e.getY());
    }

    private Point2D getScreenPosition(MouseEvent e)
    {
        java.awt.Point p = e.getLocationOnScreen();
        return new Point2D.Double(p.x, p.y);
    }

    private int getMouseButton(MouseEvent e)
    {
        int awtMouseButton = e.getButton();
        if (awtMouseButton == MouseEvent.BUTTON1)
            return org.simantics.scenegraph.g2d.events.MouseEvent.LEFT_BUTTON;
        if (awtMouseButton == MouseEvent.BUTTON2)
            return org.simantics.scenegraph.g2d.events.MouseEvent.MIDDLE_BUTTON;
        if (awtMouseButton == MouseEvent.BUTTON3)
            return org.simantics.scenegraph.g2d.events.MouseEvent.RIGHT_BUTTON;
        return awtMouseButton;
    }

    private static int getStateMask(MouseEvent e) {
        int modifiers = e.getModifiersEx();
        int stateMask = 0;
        if((modifiers & InputEvent.CTRL_DOWN_MASK) != 0)
            stateMask |= org.simantics.scenegraph.g2d.events.MouseEvent.CTRL_MASK;
        if((modifiers & InputEvent.ALT_DOWN_MASK) != 0)
            stateMask |= org.simantics.scenegraph.g2d.events.MouseEvent.ALT_MASK;
        if((modifiers & InputEvent.ALT_GRAPH_DOWN_MASK) != 0)
            stateMask |= org.simantics.scenegraph.g2d.events.MouseEvent.ALT_GRAPH_MASK;
        if((modifiers & InputEvent.SHIFT_DOWN_MASK) != 0)
            stateMask |= org.simantics.scenegraph.g2d.events.MouseEvent.SHIFT_MASK;
        return stateMask;
    }


    public void mouseEntered(MouseEvent e) {
        buttonStatus = 0;
        MouseEnterEvent me = new MouseEnterEvent(sender, e.getWhen(), MOUSE_ID, buttonStatus, getStateMask(e), getControlPosition(e), getScreenPosition(e)); 
        handleEvent(me);
    }


    public void mouseExited(MouseEvent e) {
        MouseExitEvent me = new MouseExitEvent(sender, e.getWhen(), MOUSE_ID, buttonStatus, getStateMask(e), getControlPosition(e), getScreenPosition(e));
        handleEvent(me);
    }


    public void mousePressed(MouseEvent e) {
        //System.out.println("AWT mouse pressed: " + e);
        int mouseButton = getMouseButton(e);
        if (mouseButton<=pressTime.length) pressTime[mouseButton-1] = e.getWhen();
        buttonStatus |= 1 << (mouseButton-1);

        handleEvent(new MouseButtonPressedEvent(sender, e.getWhen(), MOUSE_ID,
                buttonStatus, getStateMask(e), mouseButton, getControlPosition(e), getScreenPosition(e)));
        if (e.getClickCount() == 2) {
            handleEvent(new MouseDoubleClickedEvent(sender, e.getWhen(),
                    MOUSE_ID, buttonStatus, getStateMask(e), mouseButton, getControlPosition(e), getScreenPosition(e)));
        }
    }


    public void mouseReleased(MouseEvent e) {
        //System.out.println("AWT mouse released: " + e);
        int mouseButton = getMouseButton(e);
        long holdTime = Long.MAX_VALUE;
        if (mouseButton<=pressTime.length) 
            holdTime = e.getWhen() - pressTime[mouseButton-1];

        buttonStatus &=~ (1<<(mouseButton-1));	
        MouseButtonReleasedEvent me = new MouseButtonReleasedEvent(sender, e.getWhen(), MOUSE_ID, buttonStatus, getStateMask(e), mouseButton, holdTime, getControlPosition(e), getScreenPosition(e));
        handleEvent(me);
    }


    public void mouseDragged(MouseEvent e) {
        MouseMovedEvent me = new MouseMovedEvent(sender, e.getWhen(), MOUSE_ID, buttonStatus, getStateMask(e), getControlPosition(e), getScreenPosition(e));
        handleEvent(me);
    }


    public void mouseMoved(MouseEvent e) {
        MouseMovedEvent me = new MouseMovedEvent(sender, e.getWhen(), MOUSE_ID, buttonStatus, getStateMask(e), getControlPosition(e), getScreenPosition(e));
        handleEvent(me);
    }


    public void mouseWheelMoved(MouseWheelEvent e) {
        int wheelRotation = 0;
        if (e.getScrollType() == MouseWheelEvent.WHEEL_UNIT_SCROLL) {
            wheelRotation = -e.getUnitsToScroll();
        } else {
            wheelRotation = -e.getWheelRotation();
        }
        handleEvent(new MouseWheelMovedEvent(
                sender, e.getWhen(), MOUSE_ID, buttonStatus, getStateMask(e), getControlPosition(e), getScreenPosition(e),
                e.getScrollType(), e.getScrollAmount(), wheelRotation
                ));
    }


    public void mouseClicked(MouseEvent e) {
    }


    public void keyPressed(KeyEvent e) {
        handleEvent(
                new KeyPressedEvent(sender, e.getWhen(), e.getKeyChar(), e.getKeyCode(), e.getModifiers()));
    }


    public void keyReleased(KeyEvent e) {
        handleEvent(
                new KeyReleasedEvent(sender, e.getWhen(), e.getKeyChar(), e.getKeyCode(), e.getModifiers()));
    }


    public void keyTyped(KeyEvent e) {
//    	System.out.println("keyTyped("+e+")");
//        handleEvent(
//                new KeyTypedEvent(sender, e.getWhen(), e.getKeyChar(), e.getKeyCode(), e.getModifiers()));
    }


    public void eventDispatched(AWTEvent event) {
        if(event.getID() == KeyEvent.KEY_PRESSED) {
            keyPressed((KeyEvent)event);
        }
        if(event.getID() == KeyEvent.KEY_RELEASED) {
            keyReleased((KeyEvent)event);
        }
        if(event.getID() == KeyEvent.KEY_TYPED) {
            keyTyped((KeyEvent)event);
        }

        if(event.getID() == MouseEvent.MOUSE_MOVED) {
            mouseMoved((MouseEvent)event);
        }
        if(event.getID() == MouseEvent.MOUSE_DRAGGED) {
            mouseDragged((MouseEvent)event);
        }
        if(event.getID() == MouseEvent.MOUSE_PRESSED) {
            mousePressed((MouseEvent)event);
        }
        if(event.getID() == MouseEvent.MOUSE_RELEASED) {
            mouseReleased((MouseEvent)event);
        }
        if(event.getID() == MouseEvent.MOUSE_CLICKED) {
            mouseClicked((MouseEvent)event);
        }
        if(event.getID() == MouseEvent.MOUSE_WHEEL) {
            mouseWheelMoved((MouseWheelEvent)event);
        }
        if(event.getID() == MouseEvent.MOUSE_ENTERED) {
            mouseEntered((MouseEvent)event);
        }
        if(event.getID() == MouseEvent.MOUSE_EXITED) {
            mouseExited((MouseEvent)event);
        }
    }
}
