package org.simantics.scenegraph.g2d.events;

import java.awt.Point;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;

import org.simantics.scenegraph.g2d.G2DFocusManager;
import org.simantics.scenegraph.g2d.G2DSceneGraph;
import org.simantics.scenegraph.g2d.IG2DNode;
import org.simantics.scenegraph.utils.NodeUtil;

/**
 * Delivers AWT UI events (mouse, key, focus) to scene graph nodes that have
 * registered to it.
 * 
 * <p>
 * Mouse events are delivered as {@link SGMouseEvent} and
 * {@link SGMouseWheelEvent}. The events contain double format coordinates that
 * are in the coordinate system of the parent of the node the event is delivered
 * to.
 * 
 * @author Tuukka Lehtonen
 * @deprecated replaced with {@link NodeEventHandler}
 */
public class EventDelegator implements MouseListener, MouseMotionListener,
MouseWheelListener, KeyListener, FocusListener {

    private static final boolean                ENABLED              = true;
    private static final boolean                WARN_DEPRECATION     = false;

    protected ListenerList<MouseListener>       mouseListeners       = new ListenerList<MouseListener>(MouseListener.class);
    protected ListenerList<MouseWheelListener>  mouseWheelListeners  = new ListenerList<MouseWheelListener>(MouseWheelListener.class);
    protected ListenerList<MouseMotionListener> mouseMotionListeners = new ListenerList<MouseMotionListener>(MouseMotionListener.class);
    protected ListenerList<KeyListener>         keyListeners         = new ListenerList<KeyListener>(KeyListener.class);
    protected ListenerList<FocusListener>       focusListeners       = new ListenerList<FocusListener>(FocusListener.class);

    protected G2DSceneGraph sg = null;

    public EventDelegator(G2DSceneGraph sg) {
        this.sg = sg;
    }

    @SuppressWarnings({ "hiding" })
    public <T extends IG2DNode, MouseListener> void addMouseListener(T node) {
        mouseListeners.add((java.awt.event.MouseListener) node);
    }

    @SuppressWarnings("hiding")
    public <T extends IG2DNode, MouseListener> void removeMouseListener(T node) {
        mouseListeners.remove((java.awt.event.MouseListener) node);
    }

    @SuppressWarnings({ "hiding" })
    public <T extends IG2DNode, MouseMotionListener> void addMouseMotionListener(T node) {
        mouseMotionListeners.add((java.awt.event.MouseMotionListener) node);
    }

    @SuppressWarnings("hiding")
    public <T extends IG2DNode, MouseMotionListener> void removeMouseMotionListener(T node) {
        mouseMotionListeners.remove((java.awt.event.MouseMotionListener) node);
    }

    @SuppressWarnings({ "hiding" })
    public <T extends IG2DNode, MouseWheelListener> void addMouseWheelListener(T node) {
        mouseWheelListeners.add((java.awt.event.MouseWheelListener) node);
    }

    @SuppressWarnings("hiding")
    public <T extends IG2DNode, MouseWheelListener> void removeMouseWheelListener(T node) {
        mouseWheelListeners.remove((java.awt.event.MouseWheelListener) node);
    }

    @SuppressWarnings({ "hiding" })
    public <T extends IG2DNode, KeyListener> void addKeyListener(T node) {
        keyListeners.add((java.awt.event.KeyListener) node);
    }

    @SuppressWarnings("hiding")
    public <T extends IG2DNode, KeyListener> void removeKeyListener(T node) {
        keyListeners.remove((java.awt.event.KeyListener) node);
    }

    @SuppressWarnings({ "hiding" })
    public <T extends IG2DNode, FocusListener> void addFocusListener(T node) {
        focusListeners.add((java.awt.event.FocusListener) node);
    }

    @SuppressWarnings("hiding")
    public <T extends IG2DNode, FocusListener> void removeFocusListener(T node) {
        focusListeners.remove((java.awt.event.FocusListener) node);
    }

    @Override
    public void mouseClicked(MouseEvent event) {
        if (!ENABLED)
            return;

        Point op = event.getPoint();
        for (MouseListener l : mouseListeners.getListeners()) {
            if (WARN_DEPRECATION)
                System.out.println("DEPRECATION: " + l + " listens to " + event);
            MouseEvent e = (MouseEvent) NodeUtil.transformEvent(event, (IG2DNode) l);
            l.mouseClicked(e);
            event.translatePoint((int)(op.getX()-event.getX()), (int)(op.getY()-event.getY()));
            if (e.isConsumed())
                break;
        }
    }

    @Override
    public void mouseEntered(MouseEvent event) {
        if (!ENABLED)
            return;

        Point op = event.getPoint();
        for (MouseListener l : mouseListeners.getListeners()) {
            if (WARN_DEPRECATION)
                System.out.println("DEPRECATION: " + l + " listens to " + event);
            MouseEvent e = (MouseEvent) NodeUtil.transformEvent(event,(IG2DNode) l);
            l.mouseEntered(e);
            event.translatePoint((int)(op.getX()-event.getX()), (int)(op.getY()-event.getY()));
            if (e.isConsumed())
                break;
        }
    }

    @Override
    public void mouseExited(MouseEvent event) {
        if (!ENABLED)
            return;

        Point op = event.getPoint();
        for (MouseListener l : mouseListeners.getListeners()) {
            if (WARN_DEPRECATION)
                System.out.println("DEPRECATION: " + l + " listens to " + event);
            MouseEvent e = (MouseEvent) NodeUtil.transformEvent(event,(IG2DNode) l);
            l.mouseExited(e);
            event.translatePoint((int)(op.getX()-event.getX()), (int)(op.getY()-event.getY()));
            if (e.isConsumed())
                break;
        }
    }

    @Override
    public void mousePressed(MouseEvent event) {
        if (!ENABLED)
            return;

        G2DFocusManager.INSTANCE.clearFocus();

        Point op = event.getPoint();
        for (MouseListener l : mouseListeners.getListeners()) {
            if (WARN_DEPRECATION)
                System.out.println("DEPRECATION: " + l + " listens to " + event);
            MouseEvent e = (MouseEvent) NodeUtil.transformEvent(event,(IG2DNode) l);
            l.mousePressed(e);
            event.translatePoint((int)(op.getX()-event.getX()), (int)(op.getY()-event.getY()));
            if (e.isConsumed())
                break;
        }

        if (sg.getRootPane() != null) {
            if (G2DFocusManager.INSTANCE.getFocusOwner() == null) {
                sg.getRootPane().requestFocusInWindow();
                //sg.getRootPane().repaint(); //TODO : why repaint here? FocusOwner seems to be always null, so this causes unnecessary delays when interacting the canvas.
            }
        }
    }

    @Override
    public void mouseReleased(MouseEvent event) {
        if (!ENABLED)
            return;

        Point op = event.getPoint();
        for (MouseListener l : mouseListeners.getListeners()) {
            if (WARN_DEPRECATION)
                System.out.println("DEPRECATION: " + l + " listens to " + event);
            MouseEvent e = (MouseEvent) NodeUtil.transformEvent(event,(IG2DNode) l);
            l.mouseReleased(e);
            event.translatePoint((int)(op.getX()-event.getX()), (int)(op.getY()-event.getY()));
            if (e.isConsumed())
                break;
        }
    }

    @Override
    public void mouseDragged(MouseEvent event) {
        if (!ENABLED)
            return;

        Point op = event.getPoint();
        for (MouseMotionListener l : mouseMotionListeners.getListeners()) {
            if (WARN_DEPRECATION)
                System.out.println("DEPRECATION: " + l + " listens to " + event);
            MouseEvent e = (MouseEvent) NodeUtil.transformEvent(event,(IG2DNode) l);
            l.mouseDragged(e);
            event.translatePoint((int)(op.getX()-event.getX()), (int)(op.getY()-event.getY()));
            if (e.isConsumed())
                break;
        }
    }

    @Override
    public void mouseMoved(MouseEvent event) {
        if (!ENABLED)
            return;

        for (MouseMotionListener l : mouseMotionListeners.getListeners()) {
            if (WARN_DEPRECATION)
                System.out.println("DEPRECATION: " + l + " listens to " + event);
            MouseEvent e = (MouseEvent) NodeUtil.transformEvent(event,(IG2DNode) l);
            l.mouseMoved(e);
            if (e.isConsumed())
                break;
        }
    }

    @Override
    public void mouseWheelMoved(MouseWheelEvent event) {
        if (!ENABLED)
            return;

        Point op = event.getPoint();
        for (MouseWheelListener l : mouseWheelListeners.getListeners()) {
            if (WARN_DEPRECATION)
                System.out.println("DEPRECATION: " + l + " listens to " + event);
            MouseWheelEvent e = (MouseWheelEvent) NodeUtil.transformEvent(event, (IG2DNode) l);
            l.mouseWheelMoved(e);
            event.translatePoint((int)(op.getX()-event.getX()), (int)(op.getY()-event.getY()));
            if (e.isConsumed())
                break;
        }
    }

    @Override
    public void keyPressed(KeyEvent event) {
        if (!ENABLED)
            return;

        for (KeyListener l : keyListeners.getListeners()) {
            if (WARN_DEPRECATION)
                System.out.println("DEPRECATION: " + l + " listens to " + event);
            l.keyPressed(event);
            if (event.isConsumed())
                break;
        }
    }

    @Override
    public void keyReleased(KeyEvent event) {
        if (!ENABLED)
            return;

        for (KeyListener l : keyListeners.getListeners()) {
            if (WARN_DEPRECATION)
                System.out.println("DEPRECATION: " + l + " listens to " + event);
            l.keyReleased(event);
            if (event.isConsumed())
                break;
        }
    }

    @Override
    public void keyTyped(KeyEvent event) {
        if (!ENABLED)
            return;

        for (KeyListener l : keyListeners.getListeners()) {
            if (WARN_DEPRECATION)
                System.out.println("DEPRECATION: " + l + " listens to " + event);
            l.keyTyped(event);
            if (event.isConsumed())
                break;
        }
    }

    @Override
    public void focusGained(FocusEvent event) {
        if (!ENABLED)
            return;

        for (FocusListener l : focusListeners.getListeners()) {
            if (WARN_DEPRECATION)
                System.out.println("DEPRECATION: " + l + " listens to " + event);
            l.focusGained(event);
        }
    }

    @Override
    public void focusLost(FocusEvent event) {
        if (!ENABLED)
            return;

        for (FocusListener l : focusListeners.getListeners()) {
            if (WARN_DEPRECATION)
                System.out.println("DEPRECATION: " + l + " listens to " + event);
            l.focusLost(event);
        }
    }

    public void dispose() {
        focusListeners.clear();
        focusListeners = null;
        keyListeners.clear();
        keyListeners = null;
        mouseListeners.clear();
        mouseListeners = null;
        sg = null;
    }

}
