/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.scenegraph;

import java.io.Serializable;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * @author J-P Laine
 * 
 * @see Node
 * @see ParentNode
 */
public interface INode extends Serializable {
    public static enum Location { LOCAL, REMOTE };

    /**
     * Tell the MethodInterceptor to synchronize the given fields after the
     * method call.
     * 
     * @author J-P Laine
     */
    @Retention(RetentionPolicy.RUNTIME)
    @Target(ElementType.METHOD)
    public @interface SyncField {
        String[] value(); // Field name(s)
    }

    /**
     * Tell the MethodInterceptor to remotely call this method if we are on the
     * client side. Notice that the method type must be void, thus it cannot
     * return value.
     * 
     * @author J-P Laine
     */
    @Retention(RetentionPolicy.RUNTIME)
    @Target(ElementType.METHOD)
    public @interface ServerSide {
    }

    /**
     * Tell the MethodInterceptor to remotely call this method if we are on the
     * server side. Notice that the method type must be void, thus it cannot
     * return value.
     * 
     * @author J-P Laine
     */
    @Retention(RetentionPolicy.RUNTIME)
    @Target(ElementType.METHOD)
    public @interface ClientSide {
    }

    /**
     * Tag method as setter for a property identified by string. Basically used
     * by the automatic graph to node property synchronizer.
     * 
     * @author J-P Laine
     */
    @Retention(RetentionPolicy.RUNTIME)
    @Target(ElementType.METHOD)
    public @interface PropertySetter {
        String value(); // Property name
    }

    /**
     * 
     * @return unique node identifier
     */
    public long getId();

    /**
     * @return root node of the scene graph or <code>null</code> if this node is
     *         not part of a properly rooted scene graph hierarchy
     */
    public ParentNode<?> getRootNode();

    /**
     * @return Parent node reference or <code>null</code> if not set
     */
    public ParentNode<?> getParent();

    /**
     * Set parent node. This method is for scene graph internal use only and
     * should not be called outside the scene graph structure. This method
     * simply sets the parent node parent field, and does not affect on parent
     * node (i.e., should be called only from parent node).
     */
    public void setParent(ParentNode<?> parent);

    /**
     * Initialize node. This is called immediately after the node is added to
     * the scene graph tree through any of the node addition methods in
     * {@link ParentNode}.
     * 
     * @see ParentNode#addNode(Class)
     * @see ParentNode#addNode(String, Class)
     * @see ParentNode#getOrCreateNode(String, Class)
     */
    public void init();

    /**
     * Perform cleanup for this node and for the child nodes. Any resources
     * (including child nodes) related to this node are unusable after this
     * operation. This method is for scene graph internal use only, thus should
     * not be called outside the scene graph structure.
     */
    public void cleanup();

    /**
     * Remove this node and its children from the scene graph.
     */
    public void remove();

    /**
     * Delete this node, and move its children under its parent.
     * 
     * <p>
     * Note: this method has no effect if the node has no parent.
     */
    public void delete();

    /**
     * Append parent node for this node. The new node will be placed between
     * parent node and this node. In order to call this method the node must be
     * a child of another node, hence parent must be not null.
     */
    public <TC> TC appendParent(String id, Class<TC> nc);

    @Override
    public String toString();

    /**
     * @return simple name for this node class
     */
    public String getSimpleClassName();

}
