package org.simantics.modeling.ui.viewpoint;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;

import org.simantics.browsing.ui.model.children.ChildRule;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.db.layer0.variable.Variables.Role;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.selectionview.SelectionViewResources;

public class VariablePropertyRule implements ChildRule {

	final private ArrayList<String> requiredProperties = new ArrayList<String>();
	final private ArrayList<String> filteredProperties = new ArrayList<String>();
	
	public VariablePropertyRule(ReadGraph graph, Resource rule) throws DatabaseException {

		ModelingResources MOD = ModelingResources.getInstance(graph);
		for(Resource r : graph.getObjects(rule, MOD.ModelingBrowseContext_VariablePropertyRule_RequireProperty)) {
			String name = graph.getValue(r, Bindings.STRING);
			requiredProperties.add(name);
		}
		for(Resource r : graph.getObjects(rule, MOD.ModelingBrowseContext_VariablePropertyRule_FilterProperty)) {
			String name = graph.getValue(r, Bindings.STRING);
			filteredProperties.add(name);
		}
		
	}
	
    @Override
    public boolean isCompatible(Class<?> contentType) {
        return contentType.equals(Variable.class);
    }

    private boolean validate(ReadGraph graph, Variable parent, Variable child) throws DatabaseException {
    	
    	Resource predicate = child.getPossiblePredicateResource(graph);
    	if(predicate == null) return true;
    	
    	Layer0 L0 = Layer0.getInstance(graph);
    	if(L0.HasName.equals(predicate) || L0.HasLabel.equals(predicate)) {
            Role role = parent.getRole(graph);
            if(role == Role.PROPERTY) return false;
    	}
    	return true;
    }
    
    @Override
    public Collection<?> getChildren(ReadGraph graph, Object parent_)
            throws DatabaseException {
    	
    	Variable parent = (Variable)parent_;
    	
    	ArrayList<Variable> result = new ArrayList<Variable>();

    	Collection<Variable> properties = parent.getProperties(graph);
    	ArrayList<Resource> propertiesPredicates = new ArrayList<Resource>();
    	for (Variable property : properties) {
    	    Resource r = property.getPossiblePredicateResource(graph);
    	    if (r != null)
    	        propertiesPredicates.add(r);
    	}
    	
    	Layer0 L0 = Layer0.getInstance(graph);
    	SelectionViewResources SEL = SelectionViewResources.getInstance(graph);
    	
    	props: for(Variable property : properties) {
    		
    	    if (isUnder(graph, L0, SEL, property, propertiesPredicates)) continue props;
    	    
    	    Boolean hidden = property.getPossiblePropertyValue(graph, SEL.hidden, Bindings.BOOLEAN);
    	    if(hidden != null && hidden) continue props;
    	    
    		for(String req : requiredProperties) if(property.getPossibleProperty(graph, req) == null) continue props;

    		for(String req : filteredProperties) if(property.getName(graph).equals(req)) continue props;

    		if(!validate(graph, parent, property)) continue;
    				
    		//System.err.println("add " + property.getURI(graph));
    		
    		result.add(property);
    				
    	}
    	
    	Resource predicateResource = parent.getPossiblePredicateResource(graph);
    	if (predicateResource != null) {
        	Collection<Resource> underOfs = graph.getObjects(predicateResource, SEL.UnderOf);
        	if (!underOfs.isEmpty()) {
        	    Collection<Variable> siblings = parent.getParent(graph).getProperties(graph);
        	    for (Variable sibling : siblings) {
        	        Resource r = sibling.getPossiblePredicateResource(graph);
        	        if (r != null) {
                        if (underOfs.contains(r))
                            result.add(sibling);
        	        }
        	    }
        	}
    	}
    	
    	if(isUnderProperty(graph, parent)) {
	    	Collection<Variable> children = parent.getChildren(graph);
	    	for(Variable child : children) {
	            result.add(child);
	    	}
    	}
    	
        return result;
    }
    
    private boolean isUnderProperty(ReadGraph graph, Variable variable) throws DatabaseException {
    	Role role = variable.getRole(graph);
    	if(Role.PROPERTY.equals(role)) {
    		return true;
    	}
    	Variable parent = variable.getParent(graph);
    	if(parent == null) return false;
    	else return isUnderProperty(graph, parent);
    	
    }

    private boolean isUnder(ReadGraph graph, Layer0 L0, SelectionViewResources SEL, Variable property, ArrayList<Resource> propertiesPredicates) throws DatabaseException {
        Resource predicate = property.getPossiblePredicateResource(graph);
        if (predicate == null)
            return false;
        Collection<Resource> shownUnders = graph.getObjects(predicate, SEL.IsShownUnder);
        return !Collections.disjoint(propertiesPredicates, shownUnders);
    }
    
    @Override
    public Collection<?> getParents(ReadGraph graph, Object child)
            throws DatabaseException {
        Variable parent = ((Variable)child).getParent(graph);
        if(parent == null)
            return Collections.emptyList();
        else
            return Collections.singleton(parent);
    }

}
