package org.simantics.modeling.ui.scl.expressions;

import java.util.Collection;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.TableColumnLayout;
import org.eclipse.jface.resource.ImageRegistry;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.ColumnViewerToolTipSupport;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.ViewPart;
import org.simantics.DatabaseJob;
import org.simantics.Simantics;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.modeling.ui.Activator;
import org.simantics.scl.db.SCLExpressionTableEntry;
import org.simantics.scl.db.UsedSCLExpressionsRequest;
import org.simantics.ui.workbench.action.DefaultActions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SCLExpressionView extends ViewPart {

    private static final Logger LOGGER = LoggerFactory.getLogger(SCLExpressionView.class);

    private TableViewer tableViewer;
    private TableViewerColumn expressionColumn;
    private TableViewerColumn locationColumn;

    private ImageRegistry imageRegistry;
    private IStructuredContentProvider expressionContentProvider;
    private SCLExpressionItemFilter expressionFilter;

    public SCLExpressionView() {
        imageRegistry = Activator.getDefault().getImageRegistry();
        expressionContentProvider = ArrayContentProvider.getInstance();
    }

    @Override
    public void init(IViewSite site, IMemento memento) throws PartInitException {
        super.init(site, memento);
        IAction action = new Action("Refresh") {
            @Override
            public void run() {
                scheduleUpdateEntries();
            }
        };
        action.setImageDescriptor(imageRegistry.getDescriptor("arrow_refresh"));
        site.getActionBars().getToolBarManager().add(action);
    }

    private void createFilter(Composite parent) {
        expressionFilter = new SCLExpressionItemFilter();

        Text searchText = new Text(parent, SWT.BORDER | SWT.SEARCH);
        GridDataFactory.fillDefaults().grab(true, false).applyTo(searchText);
        searchText.addKeyListener(new KeyAdapter() {

            @Override
            public void keyReleased(KeyEvent e) {
                expressionFilter.setSearchText(searchText.getText());
                tableViewer.refresh();
            }
        });
    }

    @Override
    public void createPartControl(Composite parent) {
        parent.setLayout(new GridLayout(1, false));
        GridDataFactory.fillDefaults().grab(true, true).applyTo(parent);
        createFilter(parent);

        Composite tableParent = new Composite(parent, SWT.NONE);
        GridDataFactory.fillDefaults().grab(true, true).applyTo(tableParent);

        tableViewer = new TableViewer(tableParent,
                SWT.FULL_SELECTION | SWT.SINGLE | SWT.V_SCROLL | SWT.H_SCROLL | SWT.VIRTUAL);
        ColumnViewerToolTipSupport.enableFor(tableViewer);
        tableViewer.setContentProvider(expressionContentProvider);
        tableViewer.addFilter(expressionFilter);

        Table table = tableViewer.getTable();
        table.setHeaderVisible(true);
        table.setLinesVisible(true);

        TableColumnLayout layout = new TableColumnLayout();
        tableParent.setLayout(layout);

        expressionColumn = new TableViewerColumn(tableViewer, SWT.NONE);
        expressionColumn.getColumn().setText("Expression");
        expressionColumn.getColumn().setResizable(true);
        expressionColumn.setLabelProvider(new ColumnLabelProvider() {
            @Override
            public String getText(Object element) {
                SCLExpressionTableEntry entry = (SCLExpressionTableEntry) element;
                return entry.getContent();
            }
        });

        layout.setColumnData(expressionColumn.getColumn(), new ColumnWeightData(100));

        locationColumn = new TableViewerColumn(tableViewer, SWT.NONE);
        locationColumn.getColumn().setText("Location");
        locationColumn.getColumn().setResizable(false);
        locationColumn.setLabelProvider(new ColumnLabelProvider() {
            @Override
            public String getText(Object element) {
                SCLExpressionTableEntry entry = (SCLExpressionTableEntry) element;
                if (entry.getLocation() != null)
                    return entry.getLocation();
                return "null";
            }
        });

        layout.setColumnData(locationColumn.getColumn(), new ColumnWeightData(100));
        tableViewer.addDoubleClickListener(new IDoubleClickListener() {
            @Override
            public void doubleClick(DoubleClickEvent event) {
                IStructuredSelection selection = (IStructuredSelection) event.getSelection();
                SCLExpressionTableEntry entry = (SCLExpressionTableEntry) selection.getFirstElement();

                openResource(tableViewer.getControl().getDisplay().getActiveShell(), entry.getResource());
            }
        });
        scheduleUpdateEntries();
    }

    private void scheduleUpdateEntries() {
        Job updateJob = new DatabaseJob("Update used SCL expressions") {

            @Override
            protected IStatus run(IProgressMonitor monitor) {
                try {
                    Collection<SCLExpressionTableEntry> result = Simantics.getSession()
                            .syncRequest(new UsedSCLExpressionsRequest());
                    tableViewer.getTable().getDisplay().asyncExec(() -> {
                        tableViewer.setInput(result);
                    });
                    return Status.OK_STATUS;
                } catch (DatabaseException e) {
                    LOGGER.error("Could not update SCL expressions", e);
                    return new Status(IStatus.ERROR, Activator.PLUGIN_ID, "Could not update SCL expressions", e);
                }
            }
        };
        updateJob.schedule();
    }

    private static void openResource(Shell shell, Resource resource) {
        DefaultActions.performDefaultAction(shell, new StructuredSelection(resource));
    }

    @Override
    public void setFocus() {
        tableViewer.getControl().setFocus();
    }

    @Override
    public void dispose() {
        super.dispose();
    }

}
